package ru.yandex.direct.binlog.reader;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.github.shyiko.mysql.binlog.event.Event;
import com.github.shyiko.mysql.binlog.event.EventType;
import com.github.shyiko.mysql.binlog.event.GtidEventData;
import com.github.shyiko.mysql.binlog.event.QueryEventData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.BinlogEventConverter;
import ru.yandex.direct.mysql.BinlogRawEventServerSource;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.db.MySQLConnector;

public class BinlogStateOptimisticSnapshotter implements BinlogStateSnapshotter {
    private static final Logger logger = LoggerFactory.getLogger(BinlogStateOptimisticSnapshotter.class);

    private int serverId;
    private int maxTries;
    private Duration binlogReadTimeout;
    private int binlogMaxBufferedEvents;
    private BinlogStateSnapshotter snapshotter;

    public BinlogStateOptimisticSnapshotter(int serverId) {
        this(serverId, 5, new BinlogStateUnsafeSnapshotter());
    }

    public BinlogStateOptimisticSnapshotter(int serverId, int maxTries,
                                            BinlogStateSnapshotter snapshotter) {
        this(serverId, maxTries, Duration.ofSeconds(60), 32, snapshotter);
    }

    public BinlogStateOptimisticSnapshotter(int serverId, int maxTries,
                                            Duration binlogReadTimeout, int binlogMaxBufferedEvents, BinlogStateSnapshotter snapshotter) {
        this.serverId = serverId;
        this.maxTries = maxTries;
        this.binlogReadTimeout = binlogReadTimeout;
        this.binlogMaxBufferedEvents = binlogMaxBufferedEvents;
        this.snapshotter = snapshotter;
    }

    @Override
    public MySQLBinlogState snapshot(MySQLConnector mysql) {
        int triesLeft = maxTries;

        while (triesLeft > 0) {
            logger.info("{}: making db schema snapshot...", mysql);
            MySQLBinlogState state = snapshotter.snapshot(mysql);
            logger.info("{}: done, initial GTID set: {}", mysql, state.getGtidSet());

            // Теперь нужно убедиться, что схема не поменялась пока мы её читали
            logger.info("{}: verifying db schema consistency...", mysql);
            MySQLBinlogState state2 = snapshotter.snapshot(mysql);
            while (true) {
                if (state.getServerSchema().schemaEquals(state2.getServerSchema())) {
                    break;
                } else if (triesLeft <= 0) {
                    throw new IllegalStateException("Schema changed while we read it");
                } else {
                    triesLeft -= 1;
                    logger.warn("{}: db schema changed during snapshot", mysql);
                    state = state2;
                    state2 = snapshotter.snapshot(mysql);
                }
            }

            // Если все выглядит так, как будто схема не поменялась, это еще не значит, что она
            // не менялась на самом деле. Есть маловероятный сценарий, когда схема успела измениться
            // дважды, например, колонка переименовалась "туда и обратно" (a -> b -> a).

            // Этот сценарий можно отловить, прочитав транзакции от state.getGtidSet() до state2.getGtidSet()
            // и убедившись, что там нет DDL. Если же DDL есть - нужно падать и пробовать все с начала.

            logger.info("{}: done, looking for DDLs inside {}", mysql, state2.getGtidSet());
            if (noDDLInGtidRange(mysql, state.getGtidSet(), state2.getGtidSet())) {
                return state2;
            }

            triesLeft -= 1;

            if (triesLeft > 0) {
                logger.warn("{}: retrying to snapshot, {} tries left", mysql, triesLeft);
            }
        }

        throw new IllegalStateException("Failed to make snapshot in " + maxTries + " tries");
    }

    private Boolean noDDLInGtidRange(MySQLConnector mysql, String gtidSet1, String gtidSet2) {
        if (gtidSet1.equals(gtidSet2)) {
            return true;
        }

        try (BinlogRawEventServerSource binlog = new BinlogRawEventServerSource(
                mysql.getHost(),
                mysql.getPort(),
                mysql.getUsername(),
                mysql.getPassword(),
                serverId,
                gtidSet1,
                binlogMaxBufferedEvents
        )) {
            GtidSet unsafeGtidSet = new GtidSet(gtidSet2);
            while (true) {
                /*
                В readEvent может случится таймаут если сервер пропал/тормозит или бинлог кончился.
                Во втором случае можно было вернуть true, так как мы уже прочитали все транзакции из unsafeGtidSet
                и не нашли DDL. Но отличить второй случай от первого на практике нельзя, поэтому реагируем одинаково.
                 */
                Event event = binlog.readEvent(binlogReadTimeout.toNanos(), TimeUnit.NANOSECONDS);

                if (EventType.GTID.equals(event.getHeader().getEventType())
                        && !new GtidSet(((GtidEventData) event.getData()).getGtid()).isContainedWithin(unsafeGtidSet)
                ) {
                    return true;
                }

                if (BinlogEventConverter.looksLikeDDL(event)
                        && BinlogEventConverter.shouldApplySchemaDDL(((QueryEventData) event.getData()).getSql())) {
                    logger.warn("{}: Found DDL that might invalidate schema snapshot: {}", mysql, event.getData());
                    return false;
                }
            }
        } catch (IOException | TimeoutException exc) {
            throw new Checked.CheckedException(exc);
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(exc);
        }
    }
}
