package ru.yandex.direct.binlog.reader;

import java.util.AbstractSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;

import com.github.shyiko.mysql.binlog.GtidSet;

import ru.yandex.direct.mysql.MySQLBinlogState;

public class BinlogStateSet extends AbstractSet<NamedBinlogState> {
    private final Map<String, NamedBinlogState> stateMap;

    public BinlogStateSet() {
        stateMap = new HashMap<>();
    }

    public BinlogStateSet(NamedBinlogState record) {
        this();
        add(record);
    }

    public BinlogStateSet(Iterable<NamedBinlogState> records) {
        this();
        for (NamedBinlogState record : records) {
            add(record);
        }
    }

    public Optional<MySQLBinlogState> getBySourceName(String source) {
        return Optional.ofNullable(stateMap.get(source)).flatMap(record -> Optional.of(record.getState()));
    }

    @Nonnull
    @Override
    public Iterator<NamedBinlogState> iterator() {
        return stateMap.values().iterator();
    }

    @Override
    public int size() {
        return stateMap.size();
    }

    public boolean containsSourceName(String sourceName) {
        return stateMap.containsKey(sourceName);
    }

    @Override
    public boolean add(NamedBinlogState state) {
        if (containsSourceName(state.getSourceName())) {
            // Каждый следующий набор GTID должен быть надмножеством предыдущего

            GtidSet previous = new GtidSet(stateMap.get(state.getSourceName()).getState().getGtidSet());
            GtidSet next = new GtidSet(state.getState().getGtidSet());

            if (!previous.isContainedWithin(next)) {
                throw new IllegalStateException(
                        "GTID set diverged for source '" + state.getSourceName() + "': "
                                + previous + " is not subset of " + next
                );
            }

            // По крайней мере в текущих сценариях использования, равенство gtid set-ов - это тоже подозрительно -
            // транзакции добавились, а gtid set не изменился.
            if (previous.equals(next)) {
                throw new IllegalStateException(
                        "GTID set equals to previous for source '" + state.getSourceName() + "': " + previous
                );
            }
        }
        stateMap.put(state.getSourceName(), state);
        return true;
    }

    @Override
    public String toString() {
        return "BinlogStateSet{" + stateMap.values() + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        BinlogStateSet that = (BinlogStateSet) o;
        return Objects.equals(stateMap, that.stateMap);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), stateMap);
    }
}
