package ru.yandex.direct.binlog.reader;

import java.time.Instant;
import java.time.ZoneOffset;
import java.util.stream.Stream;

import ru.yandex.direct.mysql.BinlogEvent;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.BinlogEventType;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.Counter;

public class EnrichedEvent {
    private final String gtid;
    private final Transaction.Query query;
    private final DirectTraceInfo directTraceInfo;
    private final BinlogEvent event;
    private final int querySerial;
    private final int eventSerial;

    public EnrichedEvent(String gtid, Transaction.Query query, BinlogEvent event, int querySerial, int eventSerial) {
        this.gtid = gtid;
        this.query = query;
        // Для увеличения общности этого кода, парсинг комментариев можно было вынести и делать позже,
        // но тогда приходилось бы парсить один и тот же запрос несколько раз, потому что
        // EnrichedEvent соответствует одной изменившейся строчке в БД, а один запрос может менять сразу
        // несколько строчек.
        this.directTraceInfo = DirectTraceInfo.extractIgnoringErrors(query.getQueryString());
        this.event = event;
        this.querySerial = querySerial;
        this.eventSerial = eventSerial;
    }

    public boolean isRowsEvent() {
        return getEvent().getData() instanceof BinlogEventData.BinlogRowsEventData;
    }

    public String getRowsTable() {
        return ((BinlogEventData.BinlogRowsEventData) getEvent().getData()).getTableMap().getTable();
    }

    public String getGtid() {
        return gtid;
    }

    public Transaction.Query getQuery() {
        return query;
    }

    public DirectTraceInfo getDirectTraceInfo() {
        return directTraceInfo;
    }

    public BinlogEvent getEvent() {
        return event;
    }

    public int getQuerySerial() {
        return querySerial;
    }

    public int getEventSerial() {
        return eventSerial;
    }

    public Stream<? extends EnrichedRow> rowsStream() {
        switch (event.getType()) {
            case INSERT:
                return insertedRowsStream();
            case UPDATE:
                return updatedRowsStream();
            case DELETE:
                return deletedRowsStream();
            default:
                return Stream.empty();
        }
    }

    public Stream<EnrichedInsertRow> insertedRowsStream() {
        if (event.getType() == BinlogEventType.INSERT) {
            BinlogEventData.Insert insert = event.getData();
            Counter counter = new Counter();
            return MySQLSimpleRowsIndexed.fromSimpleRows(insert.getRows()).stream().map(row -> new EnrichedInsertRow(
                    gtid,
                    getQuery().getQueryString(),
                    getDirectTraceInfo(),
                    getQuerySerial(),
                    getEventSerial(),
                    counter.next(),
                    Instant.ofEpochMilli(event.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    insert.getTableMap().getDatabase(),
                    insert.getTableMap().getTable(),
                    insert.getTableSchema(),
                    row
            ));
        } else {
            return Stream.empty();
        }
    }

    public Stream<EnrichedUpdateRow> updatedRowsStream() {
        if (event.getType() == BinlogEventType.UPDATE) {
            BinlogEventData.Update update = event.getData();
            Counter counter = new Counter();
            return MySQLUpdateRowsIndexed.fromUpdateRows(update.getRows()).stream().map(row -> new EnrichedUpdateRow(
                    gtid,
                    getQuery().getQueryString(),
                    getDirectTraceInfo(),
                    getQuerySerial(),
                    getEventSerial(),
                    counter.next(),
                    Instant.ofEpochMilli(event.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    update.getTableMap().getDatabase(),
                    update.getTableMap().getTable(),
                    update.getTableSchema(),
                    row
            ));
        } else {
            return Stream.empty();
        }
    }

    public Stream<EnrichedDeleteRow> deletedRowsStream() {
        if (event.getType() == BinlogEventType.DELETE) {
            BinlogEventData.Delete delete = event.getData();
            Counter counter = new Counter();
            return MySQLSimpleRowsIndexed.fromSimpleRows(delete.getRows()).stream().map(row -> new EnrichedDeleteRow(
                    gtid,
                    getQuery().getQueryString(),
                    getDirectTraceInfo(),
                    getQuerySerial(),
                    getEventSerial(),
                    counter.next(),
                    Instant.ofEpochMilli(event.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    delete.getTableMap().getDatabase(),
                    delete.getTableMap().getTable(),
                    delete.getTableSchema(),
                    row
            ));
        } else {
            return Stream.empty();
        }
    }
}
