package ru.yandex.direct.binlog.reader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import ru.yandex.direct.mysql.BinlogEvent;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.BinlogEventType;
import ru.yandex.direct.utils.Counter;

public class Transaction {
    private final String gtid;
    private final List<Query> queries;
    private int queryStartIndex = 0;
    private int eventStartIndex = 0;

    public Transaction(String gtid, List<Query> queries) {
        this.gtid = gtid;
        this.queries = queries;
    }

    public String getGtid() {
        return gtid;
    }

    public List<Query> getQueries() {
        return queries;
    }

    public int getQueryStartIndex() {
        return queryStartIndex;
    }

    public int getEventStartIndex() {
        return eventStartIndex;
    }

    public Transaction withQueryStartIndex(int queryStartIndex) {
        this.queryStartIndex = queryStartIndex;
        return this;
    }

    public Transaction withEventStartIndex(int eventStartIndex) {
        this.eventStartIndex = eventStartIndex;
        return this;
    }

    public Stream<EnrichedEvent> getEnrichedEvents() {
        Counter eventCounter = new Counter(eventStartIndex);
        Counter queryCounter = new Counter(queryStartIndex);
        return queries.stream().flatMap(
                query -> {
                    int querySerial = queryCounter.next();
                    return query.getEvents().stream().map(
                            event -> new EnrichedEvent(
                                    gtid,
                                    query,
                                    event,
                                    querySerial,
                                    eventCounter.next()
                            )
                    );
                }
        );
    }

    public static class Builder {
        private String gtid;
        private List<Query> queries;
        private QueryBuilder currentQuery;
        private int queryStartIndex = 0;
        private int eventStartIndex = 0;

        Builder(String gtid) {
            this.gtid = gtid;
            this.queries = new ArrayList<>();
            this.currentQuery = null;
        }

        Builder(String gtid, int queryStartIndex, int eventStartIndex) {
            this(gtid);
            this.queryStartIndex = queryStartIndex;
            this.eventStartIndex = eventStartIndex;
        }

        public String getGtid() {
            return gtid;
        }

        void initNewQuery(BinlogEvent rowsQuery) {
            if (!BinlogEventType.ROWS_QUERY.equals(rowsQuery.getType())) {
                throw new IllegalStateException("Expected event of type ROWS_QUERY, got: " + rowsQuery.getType() + " " +
                        "(gtid=" + gtid + ")");
            }
            if (currentQuery != null) {
                queries.add(currentQuery.build());
            }
            currentQuery = new QueryBuilder(rowsQuery);
        }

        QueryBuilder getCurrentQuery() {
            if (currentQuery == null) {
                throw new IllegalStateException("Transaction has no queries yet (gtid=" + gtid + ")");
            }
            return currentQuery;
        }

        QueryBuilder getOrCreateQuery() {
            if (currentQuery == null) {
                currentQuery = new QueryBuilder("");
            }
            return currentQuery;
        }

        Transaction commit() {
            if (currentQuery != null) {
                queries.add(currentQuery.build());
            }
            return new Transaction(gtid, Collections.unmodifiableList(queries))
                    .withQueryStartIndex(queryStartIndex)
                    .withEventStartIndex(eventStartIndex);
        }
    }

    static class QueryBuilder {
        private final BinlogEvent rowsQueryEvent;
        private final List<BinlogEvent> events;
        private final String query;

        QueryBuilder(String query) {
            this.query = query;
            this.rowsQueryEvent = null;
            this.events = new ArrayList<>();
        }

        QueryBuilder(BinlogEvent rowsQueryEvent) {
            this.query = null;
            this.rowsQueryEvent = rowsQueryEvent;
            this.events = new ArrayList<>();
        }

        void addEvent(BinlogEvent event) {
            events.add(event);
        }

        int getSize() {
            return events.size();
        }

        Query build() {
            if (rowsQueryEvent != null) {
                return new Query(rowsQueryEvent, Collections.unmodifiableList(events));
            } else {
                return new Query(query, Collections.unmodifiableList(events));
            }
        }
    }

    public static class Query {
        private final BinlogEvent rowsQueryEvent;
        private final List<BinlogEvent> events;
        private final String query;

        public Query(BinlogEvent rowsQueryEvent, List<BinlogEvent> events) {
            this.rowsQueryEvent = rowsQueryEvent;
            this.query = null;
            this.events = events;
        }

        public Query(String query, List<BinlogEvent> events) {
            this.query = query;
            this.rowsQueryEvent = null;
            this.events = events;
        }

        public String getQueryString() {
            return rowsQueryEvent != null
                    ? ((BinlogEventData.RowsQuery) rowsQueryEvent.getData()).getData().getQuery()
                    : query;
        }

        public List<BinlogEvent> getEvents() {
            return events;
        }
    }
}
