package ru.yandex.direct.binlog.reader;

import java.time.Duration;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.direct.mysql.BinlogEvent;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.BinlogEventType;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

public class TransactionReader {
    private static final Duration INSIDE_TRANSACTION_TIMEOUT = Duration.ofSeconds(180);
    private final BinlogReader reader;
    // Максимальное количество событий для чтения в транзакции. Если их больше, то завершает чтение транзакции,
    // не дожидаясь COMMIT. Далее продолжает с события, на котором остановились в прошлый раз, то есть транзакция
    // делится на чанки размером maxEventsPerTransaction. Если null, то ограничения нет, и транзакция читается целиком.
    private Integer maxEventsPerTransaction;
    private int eventsCount = 0;
    private int queriesCount = 0;
    private int eventsCountOfQuery = 0;
    private String lastProcessedGtid;
    private BinlogEvent lastProcessedQueryEvent;
    private BinlogEvent lastProcessedEvent;
    // Отправить запрос в топик для запросов.
    // Нужен только для ситуации, когда прошлое чтение транзакции закончилось между запросами, а не между событиями.
    // То есть запрос начал читаться, но ни одного события не было отправлено.
    private boolean needWriteQueryOnFirstQuery = false;

    public TransactionReader(BinlogReader reader) {
        this.reader = reader;
    }

    public TransactionReader withMaxEventsPerTransaction(Integer maxEventsPerTransaction) {
        this.maxEventsPerTransaction = maxEventsPerTransaction;
        return this;
    }

    public NamedBinlogState getState() {
        return reader.getState();
    }

    public int getEventsCount() {
        return this.eventsCount;
    }

    public boolean getNeedWriteQueryOnFirstQuery() {
        return needWriteQueryOnFirstQuery;
    }

    /**
     * Может читать дольше timeout в следующих случаях:
     * <ul>
     * <li>Если был получен DDL, то процедура получения транзакции начинается заново.</li>
     * <li>Если канал узкий и не удаётся за вменяемое время прочитать всю транзакцию начиная с BEGIN и заканчивая
     * COMMIT</li>
     * <li>Если был получен ROLLBACK, то процедура получения транзакции начинается заново.</li>
     * </ul>
     */
    @SuppressWarnings("squid:S1151") // Больше пяти строчек в case BEGIN
    public StateBound<Transaction> readTransaction(Duration timeout) throws InterruptedException {
        while (true) {
            StateBound<TransactionOrDdl> transactionOrDdl = readTransactionOrDdl(timeout);
            if (transactionOrDdl.getData().getTransaction() != null) {
                return new StateBound<>(transactionOrDdl.getStateSet(), transactionOrDdl.getData().getTransaction());
            }
        }
    }

    // Если приложение остановилось, не отправив транзакцию до конца, нам надо пропустить уже отправленные в lb события,
    // и продолжить чтение.
    public void skipFirstEventsFromTransaction(Duration timeout, int eventsToSkip) throws InterruptedException {
        if (eventsToSkip <= 0) {
            return;
        }

        BinlogEvent event = reader.readEvent(timeout);
        if (event.getType() != BinlogEventType.BEGIN) {
            throw new IllegalStateException("Expected BEGIN, got: " + event.getType());
        }
        lastProcessedGtid = event.getData().getGtid();
        queriesCount = 0;
        eventsCount = 0;
        eventsCountOfQuery = 0;

        while (eventsCount < eventsToSkip) {
            try {
                event = reader.readEvent(INSIDE_TRANSACTION_TIMEOUT);
            } catch (RuntimeTimeoutException exc) {
                throw new StuckTransaction(lastProcessedGtid, INSIDE_TRANSACTION_TIMEOUT);
            }

            if (!event.getData().getGtid().equals(lastProcessedGtid)) {
                throw new IllegalStateException("Bad state, transaction " + lastProcessedGtid +
                        " contains less than " + eventsToSkip + " events with data.");
            }
            if (event.getType() == BinlogEventType.ROWS_QUERY) {
                ++queriesCount;
                lastProcessedQueryEvent = event;
                eventsCountOfQuery = 0;
            }

            if (event.getType().equals(BinlogEventType.INSERT) || event.getType().equals(BinlogEventType.UPDATE) ||
                    event.getType().equals(BinlogEventType.DELETE)) {
                ++eventsCount;
                ++eventsCountOfQuery;
            }
        }

        // Читаем событие, на котором остановились в прошлый раз
        lastProcessedEvent = reader.readEvent(INSIDE_TRANSACTION_TIMEOUT);
    }

    public StateBound<TransactionOrDdl> readTransactionOrDdl(Duration timeout) throws InterruptedException {
        while (true) {
            // Если счетчик событий не нулевой, то значит мы не дочитали прошлую транзакцию.
            if (eventsCount == 0) {
                BinlogEvent event = reader.readEvent(timeout);
                switch (event.getType()) {
                    case BEGIN:
                        BinlogEventData.Begin begin = event.getData();
                        lastProcessedGtid = begin.getGtid();
                        break;

                    case DDL:
                        return new StateBound<>(new BinlogStateSet(getState()),
                                new TransactionOrDdl(null, event));

                    default:
                        throw new IllegalStateException("Expected BEGIN, got: " + event.getType());
                }
            }

            Transaction.Builder transactionBuilder = new Transaction.Builder(lastProcessedGtid,
                    Math.max(queriesCount - 1, 0), eventsCount);

            needWriteQueryOnFirstQuery = false;
            if (lastProcessedEvent != null) {
                transactionBuilder.initNewQuery(lastProcessedQueryEvent);
                if (eventsCountOfQuery == 0) {
                    needWriteQueryOnFirstQuery = true;
                }
            }

            Optional<Transaction> transaction = completeTransaction(transactionBuilder, timeout);
            if (transaction.isPresent()) {
                return new StateBound<>(new BinlogStateSet(getState()),
                        new TransactionOrDdl(transaction.get(), null));
            }
        }
    }

    private Optional<Transaction> completeTransaction(Transaction.Builder transactionBuilder, Duration timeout)
            throws InterruptedException {
        int eventsCountOnStart = eventsCount;

        while (true) {
            BinlogEvent event;
            try {
                if (lastProcessedEvent != null) {
                    event = lastProcessedEvent;
                    lastProcessedEvent = null;
                } else {
                    event = reader.readEvent(INSIDE_TRANSACTION_TIMEOUT);
                }
            } catch (RuntimeTimeoutException exc) {
                throw new StuckTransaction(transactionBuilder.getGtid(), INSIDE_TRANSACTION_TIMEOUT);
            }

            switch (event.getType()) {
                case BEGIN:
                    BinlogEventData.Begin begin = event.getData();
                    throw new IllegalStateException(
                            "Transaction begin (gtid=" + begin.getGtid() +
                                    "), while another transaction in progress (gtid=" + transactionBuilder.getGtid()
                                    + ")"
                    );

                case ROWS_QUERY:
                    ++queriesCount;
                    lastProcessedQueryEvent = event;
                    eventsCountOfQuery = 0;
                    transactionBuilder.initNewQuery(event);
                    break;

                case INSERT:
                case UPDATE:
                case DELETE:
                    // Событие не влезает в лимит, отправляем то, что есть, и сохраняем данное событие,
                    // чтобы потом продолжить с него.
                    if (maxEventsPerTransaction != null && eventsCount - eventsCountOnStart >= maxEventsPerTransaction) {
                        lastProcessedEvent = event;
                        return Optional.of(transactionBuilder.commit());
                    }

                    ++eventsCount;
                    ++eventsCountOfQuery;
                    transactionBuilder.getOrCreateQuery().addEvent(event);
                    break;

                // Предполагаем, что ROLLBACK'ов нет как минимум для длинных транзакций, которые надо чанковать,
                // иначе в базу отправятся изменения, которые по факту не произошли.
                case ROLLBACK:
                    resetIntermediateState();
                    // BinlogReader возвращает фейковый ROLLBACK, если произошел реконнект к mysql.
                    // Нам нужно снова пропустить уже отправленные события из транзакции.
                    if (event.getTimestamp() == 0) {
                        skipFirstEventsFromTransaction(timeout, eventsCountOnStart);
                    }
                    return Optional.empty();

                case COMMIT:
                    resetIntermediateState();
                    return Optional.of(handleCommit(transactionBuilder, event));

                case DML:
                    break;

                default:
                    throw new IllegalStateException("Unexpected event type: " + event.getType());
            }
        }
    }

    private void resetIntermediateState() {
        queriesCount = 0;
        eventsCount = 0;
        eventsCountOfQuery = 0;
        lastProcessedQueryEvent = null;
        lastProcessedEvent = null;
        lastProcessedGtid = null;
    }

    private Transaction handleCommit(Transaction.Builder transactionBuilder, BinlogEvent event) {
        BinlogEventData.Commit commit = event.getData();
        if (!commit.getGtid().equals(transactionBuilder.getGtid())) {
            throw new IllegalStateException(
                    "Unexpected commit for gtid=" + commit.getGtid() +
                            ", expected gtid=" + transactionBuilder.getGtid()
            );
        }
        return transactionBuilder.commit();
    }

    public static class TransactionOrDdl {
        @Nullable
        private final Transaction transaction;

        @Nullable
        private final BinlogEvent ddl;

        public TransactionOrDdl(@Nullable Transaction transaction, @Nullable BinlogEvent ddl) {
            Preconditions.checkArgument((transaction == null) != (ddl == null),
                    "Either transaction or ddl should be null");
            this.transaction = transaction;
            this.ddl = ddl;
        }

        @Nullable
        public Transaction getTransaction() {
            return transaction;
        }

        @Nullable
        public BinlogEvent getDdl() {
            return ddl;
        }
    }

    /**
     * Если в mysql успешно произошла транзакция, то он пишет в бинлог множество событий в последовательности
     * BEGIN, ROWS_QUERY, ..., COMMIT. В идеальном случае все эти события должны идти мгновенно друг за другом,
     * но существует ничтожная вероятность, что соединение порвётся аккуратно между событиями.
     * Текущая реализация TransactionReader не позволяет корректно обработать такую ситуацию (DIRECT-80970),
     * поэтому бросается ошибка, при получении которой лучше сделать полноценное восстановление из стейта
     * без всяких оптимизированных реконнектов.
     */
    public static class StuckTransaction extends RuntimeException {
        StuckTransaction(String gtid, Duration timeout) {
            super(String.format("Stuck for %s while read transaction with gtid %s", timeout, gtid));
        }
    }
}
