package ru.yandex.direct.blackbox.client;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.PreDestroy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.solomon.SolomonResponseMonitorStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.AliasesParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.EmailsParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.PhoneAttributesParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxLoginException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.misc.ip.IpAddress;

import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNKNOWN;

/**
 * Обертка над {@link Blackbox2}, отправляющая метрики в Соломон
 */
public class BlackboxClient {

    private static final String EXTERNAL_SYSTEM = "blackbox";
    private static final String METHOD_USER_INFO = "userInfo";
    private static final String METHOD_USER_INFO_BULK = "userInfoBulk";
    private static final String METHOD_USER_TICKET = "userTicket";
    private static final String METHOD_OAUTH = "oAuth";
    private static final String METHOD_SESSION_ID = "sessionId";

    private static final SolomonExternalSystemMonitorService MONITOR_SERVICE = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(METHOD_USER_INFO_BULK, METHOD_USER_INFO, METHOD_USER_TICKET, METHOD_OAUTH, METHOD_SESSION_ID)
    );

    private final Blackbox2 blackbox2;

    public BlackboxClient(Blackbox2 blackbox2) {
        this.blackbox2 = blackbox2;
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            PassportUid passportUid,
            List<String> dbFields,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, passportUid, null, dbFields, null, null, null, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            String login,
            List<String> dbFields,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, null, login, dbFields, null, null, null, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            List<Integer> attributes,
            String login,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, null, login, null, attributes, null, null, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            PassportUid passportUid,
            AliasesParameterValue aliases,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, passportUid, null, null, null, aliases, null, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            String login,
            AliasesParameterValue aliases,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, null, login, null, null, aliases, null, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            PassportUid passportUid,
            EmailsParameterValue emails,
            String tvmServiceTicket
    ) throws BlackboxException {
        return userInfo(userIp, passportUid, null, null, null, null, emails, tvmServiceTicket);
    }

    public BlackboxCorrectResponse userInfo(
            IpAddress userIp,
            PassportUid passportUid,
            String login,
            List<String> dbFields,
            List<Integer> attributes,
            AliasesParameterValue aliases,
            EmailsParameterValue emails,
            String tvmServiceTicket
    ) throws BlackboxException {
        try {
            var response = blackbox2.query().userInfo(
                    userIp,
                    passportUid == null ? Option.empty() : Option.of(passportUid),
                    login == null ? Option.empty() : Option.of(login),
                    Option.empty(),
                    dbFields == null ? Cf.list() : dbFields,
                    attributes == null ? Cf.list() : Cf.x(attributes),
                    emails == null ? Option.empty() : Option.of(emails),
                    aliases == null ? Option.empty() : Option.of(aliases),
                    false,
                    Option.empty(),
                    false,
                    false,
                    Option.of(tvmServiceTicket)
            );
            MONITOR_SERVICE.write(METHOD_USER_INFO, STATUS_2XX);
            return response;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_USER_INFO, monitorStatus);
            throw e;
        }
    }

    public Map<PassportUid, BlackboxAbstractResponse> userInfoBulk(
            IpAddress userIp,
            List<PassportUid> uids,
            List<String> dbFields,
            Optional<EmailsParameterValue> emails,
            Optional<AliasesParameterValue> aliases,
            boolean regName,
            Optional<List<PhoneAttributesParameterValue>> phoneAttributes,
            String tvmServiceTicket
    ) throws BlackboxException {
        try {
            var response = blackbox2.query().userInfoBulk(userIp, uids, dbFields, emails, aliases, regName,
                    phoneAttributes, tvmServiceTicket);
            MONITOR_SERVICE.write(METHOD_USER_INFO_BULK, STATUS_2XX);
            return response;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_USER_INFO_BULK, monitorStatus);
            throw e;
        }
    }

    public BlackboxCorrectResponse userTicket(
            String userTicket,
            AliasesParameterValue aliases,
            String tvmServiceTicket
    ) throws BlackboxException {
        try {
            var response = blackbox2.query().userTicket(userTicket, aliases, tvmServiceTicket);
            MONITOR_SERVICE.write(METHOD_USER_TICKET, STATUS_2XX);
            return response;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_USER_TICKET, monitorStatus);
            throw e;
        }
    }

    public BlackboxCorrectResponse oAuth(
            IpAddress userIp,
            String token,
            List<String> dbFields,
            List<Integer> attributes,
            Optional<EmailsParameterValue> emails,
            Optional<AliasesParameterValue> aliases,
            boolean getUserTicket,
            String tvmServiceTicket
    ) throws BlackboxException {
        try {
            var response = blackbox2.query().oAuth(userIp, token, dbFields, attributes, emails, aliases, getUserTicket,
                    tvmServiceTicket);
            MONITOR_SERVICE.write(METHOD_OAUTH, STATUS_2XX);
            return response;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_OAUTH, monitorStatus);
            throw e;
        }
    }

    public BlackboxCorrectResponse sessionId(
            IpAddress userIp,
            String sessionId,
            String host,
            List<String> dbFields,
            boolean renew,
            Optional<String> sslSessionId,
            boolean getUserTicket,
            String tvmServiceTicket
    ) throws BlackboxException {
        try {
            var response = blackbox2.query().sessionId(userIp, sessionId, host, dbFields, renew, sslSessionId,
                    getUserTicket, tvmServiceTicket);
            MONITOR_SERVICE.write(METHOD_SESSION_ID, STATUS_2XX);
            return response;
        } catch (RuntimeException e) {
            var monitorStatus = guessStatusByException(e);
            MONITOR_SERVICE.write(METHOD_SESSION_ID, monitorStatus);
            throw e;
        }
    }

    private static SolomonResponseMonitorStatus guessStatusByException(RuntimeException exception) {
        if (exception instanceof BlackboxSessionIdException
                || exception instanceof BlackboxLoginException
                || exception instanceof BlackboxOAuthException) {
            return STATUS_4XX;
        }
        if (exception instanceof BlackboxException) {
            return STATUS_5XX;
        }
        return STATUS_UNKNOWN;
    }

    @PreDestroy
    private void destroy() {
        blackbox2.destroy();
    }

}

