package ru.yandex.direct.bsauction;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Joiner;

import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.utils.net.FastUrlBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class BsRequest<T extends BsRequestPhrase> {

    private static final Joiner VALUES_JOINER = Joiner.on('\n');
    private static final String DEFAULT_DOMAIN = "yandex.ru";
    private static final int DEFAULT_PAGE_ID = 24;

    // номер опции в БК для трафаретных торгов РМП и ТГО с таргетином на устройства ( BSSERVER-4865 BSRELEASE-129507 )
    private static final String TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID_VALUE = "13";

    private static final String PARAM_NAME_RANK_OPTION_ID = "rank-option-id";
    private static final String PARAM_NAME_TARGET_DEVICE_GROUP = "target-device-group";
    private static final String PARAM_NAME_TARGET_DETAILED_DEVICE_TYPE = "target-detailed-device-type";
    private static final String PARAM_NAME_MAIN_CTR_PRED_ID = "main-ctr-pred-id";
    private static final String PARAM_NAME_MCBANNER = "mcbanner";

    private int pageId = DEFAULT_PAGE_ID;       // pages.pageID. Используется как часть урла торгов.
    private long orderId;                       // banner.OrderID
    private List<Long> regionIds;               // banner.geo
    private Boolean noExtendedGeotargeting;     // banner.no_extended_geotargeting
    private Currency currency;                  // banner.currency, convert to ISO code (RUR, USD, etc)
    private String domain = DEFAULT_DOMAIN;     // default yandex.ru, if banner.filter_domain && banner.phone,
    // then domain from phone
    private boolean timetable = false;          // banner.fairAuction
    private String bannerHead;                  // banner.title or banner.phrases[0]
    private String bannerBody;                  // banner.body or null
    private BidCalculationMethod bidCalcMethod; // метода расчета ставки на аукционе
    private boolean onlyMobilePages = false;
    private Integer mainCtrPredictionId;        // формула для рассчетов cpm и т.д.
    // (https://st.yandex-team.ru/BSSERVER-9994#5d9b3b64a2b79e001d88f6a3)

    private boolean targetDeviceDesktop = false;
    private boolean targetDevicePhone = false;
    private boolean targetDeviceTablet = false;

    private List<String> targetDeviceGroups = new ArrayList<>(); // 0 - desktop, 1 - phone, 2 - tablet
    private BsTargetDetailedDeviceType targetDetailedDeviceType;

    private List<T> phrases;

    private boolean mcBanner = false;
    private Integer rankOptionId;

    public BsRequest<T> withPageId(int pageId) {
        this.pageId = pageId;
        return this;
    }

    public BsRequest<T> withOrderId(long orderId) {
        this.orderId = orderId;
        return this;
    }

    public BsRequest<T> withRegionIds(List<Long> regionIds) {
        this.regionIds = regionIds;
        return this;
    }

    public BsRequest<T> withNoExtendedGeotargeting(Boolean noExtendedGeotargeting) {
        this.noExtendedGeotargeting = noExtendedGeotargeting;
        return this;
    }

    public BsRequest<T> withCurrency(Currency currency) {
        this.currency = currency;
        return this;
    }

    public int getPageId() {
        return pageId;
    }

    public Currency getCurrency() {
        return this.currency;
    }

    public BsRequest<T> withDomain(String domain) {
        this.domain = domain;
        return this;
    }

    public BsRequest<T> withTimetable(boolean timetable) {
        this.timetable = timetable;
        return this;
    }

    public BsRequest<T> withBannerHead(String bannerHead) {
        this.bannerHead = bannerHead;
        return this;
    }

    public BsRequest<T> withBannerBody(String bannerBody) {
        this.bannerBody = bannerBody;
        return this;
    }

    public BsRequest<T> withBidCalcMethod(BidCalculationMethod bidCalcMethod) {
        this.bidCalcMethod = bidCalcMethod;
        return this;
    }

    public BsRequest<T> withOnlyMobilePages(boolean onlyMobilePages) {
        this.onlyMobilePages = onlyMobilePages;
        return this;
    }

    public BsRequest<T> withTargetDeviceDesktop(boolean targetDeviceDesktop) {
        this.targetDeviceDesktop = targetDeviceDesktop;
        return this;
    }

    public BsRequest<T> withTargetDevicePhone(boolean targetDevicePhone) {
        this.targetDevicePhone = targetDevicePhone;
        return this;
    }

    public BsRequest<T> withTargetDeviceTablet(boolean targetDeviceTablet) {
        this.targetDeviceTablet = targetDeviceTablet;
        return this;
    }

    public BsRequest<T> withTargetDetailedDeviceType(BsTargetDetailedDeviceType targetDetailedDeviceType) {
        this.targetDetailedDeviceType = targetDetailedDeviceType;
        return this;
    }

    public BsRequest<T> withPhrases(List<T> phrases) {
        this.phrases = phrases;
        return this;
    }

    public BsRequest<T> withMainCtrPredictionId(int mainCtrPredictionId) {
        this.mainCtrPredictionId = mainCtrPredictionId;
        return this;
    }

    public BsRequest<T> withMcBanner(boolean mcBanner) {
        this.mcBanner = mcBanner;
        return this;
    }

    public BsRequest<T> withRankOptionId(Integer rankOptionId) {
        this.rankOptionId = rankOptionId;
        return this;
    }

    public long getOrderId() {
        return orderId;
    }

    public List<Long> getRegionIds() {
        return regionIds;
    }

    public Boolean getNoExtendedGeotargeting() {
        return noExtendedGeotargeting;
    }

    public String getDomain() {
        return domain;
    }

    public boolean isTimetable() {
        return timetable;
    }

    public String getBannerHead() {
        return bannerHead;
    }

    public String getBannerBody() {
        return bannerBody;
    }

    public BidCalculationMethod getBidCalcMethod() {
        return bidCalcMethod;
    }

    public Integer getMainCtrPredictionId() {
        return mainCtrPredictionId;
    }

    public boolean isOnlyMobilePages() {
        return onlyMobilePages;
    }

    public List<T> getPhrases() {
        return phrases;
    }

    public static String getParamNameRankOptionId() {
        return PARAM_NAME_RANK_OPTION_ID;
    }

    public static String getParamNameTargetDeviceGroup() {
        return PARAM_NAME_TARGET_DEVICE_GROUP;
    }

    public static String getParamNameTargetDetailedDeviceType() {
        return PARAM_NAME_TARGET_DETAILED_DEVICE_TYPE;
    }

    public static String getTrafaretMobileAndDeviceTargetingRankOptionIdValue() {
        return TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID_VALUE;
    }

    public boolean isMcBanner() {
        return mcBanner;
    }

    public Integer getRankOptionId() {
        return rankOptionId;
    }

    public FastUrlBuilder getUrlBuilder(String bsUrl) {
        FastUrlBuilder builder = new FastUrlBuilder(bsUrl);

        builder.addPath(pageId);

        if (regionIds == null || regionIds.isEmpty()) {
            builder.addParam("reg-id", '0');
        } else {
            builder.addParam("reg-id", VALUES_JOINER.join(regionIds));
        }

        if (noExtendedGeotargeting != null) {
            builder.addParam("no-extended-geotargeting", noExtendedGeotargeting ? '1' : '0');
        }

        checkNotNull(currency, "Currency is required for BsRequest");
        if (currency.getIsoNumCode() != null) {
            builder.addParam("currency", currency.getIsoNumCode());
        }

        builder.addParam("domain", nvl(domain, DEFAULT_DOMAIN));

        if (timetable) {
            builder.addParam("timetable", '1');
        }

        builder.addParam("vcg-auction", '1');

        if (bannerHead != null) {
            builder.addParam("ban-head", bannerHead);
        }

        if (bannerBody != null) {
            builder.addParam("ban-body", bannerBody);
        }

        if (bidCalcMethod != null) {
            builder.addParam("operation", bidCalcMethod.operation());
        }

        if (onlyMobilePages) {
            builder.addParam("only-mobile-pages", '1');
        }

        if (!(targetDeviceDesktop && targetDevicePhone && targetDeviceTablet)) {
            if (targetDeviceDesktop) {
                targetDeviceGroups.add("0");
            }

            if (targetDevicePhone) {
                targetDeviceGroups.add("1");
            }

            if (targetDeviceTablet) {
                targetDeviceGroups.add("2");
            }
        }

        boolean needTrafaretMobileRankOptionId = false;

        if (targetDeviceGroups.size() > 0) {
            needTrafaretMobileRankOptionId = true;
            builder.addParam(PARAM_NAME_TARGET_DEVICE_GROUP, VALUES_JOINER.join(targetDeviceGroups));
        }

        if (targetDetailedDeviceType != null) {
            needTrafaretMobileRankOptionId = true;
            builder.addParam(PARAM_NAME_TARGET_DETAILED_DEVICE_TYPE, targetDetailedDeviceType.toString());
        }

        if (mcBanner) {
            needTrafaretMobileRankOptionId = true;
        }

        if (needTrafaretMobileRankOptionId) {
            builder.addParam(
                    PARAM_NAME_RANK_OPTION_ID, TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID_VALUE
            );
        }

        if (mainCtrPredictionId != null) {
            builder.addParam(PARAM_NAME_MAIN_CTR_PRED_ID, mainCtrPredictionId);
        }

        if (phrases != null) {
            for (int i = 0; i < phrases.size(); i++) {
                BsRequestPhrase phrase = phrases.get(i);
                builder.addParam("target", phrase.getQueryStringValue(i, orderId));
            }
        }

        if (mcBanner) {
            builder.addParam(PARAM_NAME_MCBANNER, '1');
        }

        return builder;
    }

    @Override
    public String toString() {
        return "BsRequest{" +
                "pageId=" + pageId +
                ", orderId=" + orderId +
                ", phrases=" + phrases +
                ", domain='" + domain + '\'' +
                ", regionIds=" + regionIds +
                ", bannerHead='" + bannerHead + '\'' +
                ", bannerBody='" + bannerBody + '\'' +
                ", noExtendedGeotargeting=" + noExtendedGeotargeting +
                ", currency=" + currency +
                ", timetable=" + timetable +
                ", bidCalcMethod=" + bidCalcMethod +
                ", onlyMobilePages=" + onlyMobilePages +
                ", mainCtrPredictionId=" + mainCtrPredictionId +
                ", targetDeviceDesktop=" + targetDeviceDesktop +
                ", targetDevicePhone=" + targetDevicePhone +
                ", targetDeviceTablet=" + targetDeviceTablet +
                ", targetDeviceGroups=" + targetDeviceGroups +
                ", targetDetailedDeviceType=" + targetDetailedDeviceType +
                ", mcbanner=" + mcBanner +
                '}';
    }
}
