package ru.yandex.direct.bsauction;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.bsauction.parser.BsTrafaretParser;
import ru.yandex.direct.bsauction.parser.FullBsTrafaretParser;
import ru.yandex.direct.bsauction.parser.ParsableBsTrafaretRequest;
import ru.yandex.direct.bsauction.parser.PositionalBsTrafaretParser;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.MicroMoneyConverter;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.net.FastUrlBuilder;


/**
 * Клиент к Торгам, использующим формат ответа с данными по трафаретам
 */
public class BsTrafaretClient {
    private static final Logger logger = LoggerFactory.getLogger(BsTrafaretClient.class);
    private static final String HOST_HEADER = "Host";

    private final String bsUrl;
    private final String bsHostHeader;
    private final ParallelFetcherFactory fetcherFactory;
    private final Supplier<Duration> softTimeoutSupplier;

    public BsTrafaretClient(ParallelFetcherFactory fetcherFactory, String bsUrl, String bsHostHeader,
                            Supplier<Duration> softTimeoutSupplier) {
        this.fetcherFactory = fetcherFactory;
        this.bsUrl = bsUrl;
        this.bsHostHeader = bsHostHeader;
        this.softTimeoutSupplier = softTimeoutSupplier;
    }

    /**
     * Маппинг trafficVolume на старые позиции
     *
     * @see <a href="https://wiki.yandex-team.ru/users/maxlog/work/new-auction/questions/#mappingpozicijj">wiki</a>
     */
    public static final ImmutableMap<Place, Long> PLACE_TRAFARET_MAPPING = ImmutableMap.<Place, Long>builder()
            .put(Place.PREMIUM1, 100_0000L)
            .put(Place.PREMIUM2, 85_0000L)
            .put(Place.PREMIUM3, 73_0000L)
            .put(Place.PREMIUM4, 62_0000L)
            .put(Place.GUARANTEE1, 8_5000L)
            .put(Place.GUARANTEE2, 7_2000L)
            .put(Place.GUARANTEE3, 6_7000L)
            .put(Place.GUARANTEE4, 5_0000L)
            .build();


    private static <R extends BsRequestPhrase, M extends ResponsePhraseWithId> BsResponse<R, M> convertResult(
            Result<IdentityHashMap<R, M>> result) {
        if (result.getSuccess() != null) {
            return BsResponse.success(result.getSuccess());
        }
        return BsResponse.failure(result.getErrors());
    }

    <R extends BsRequestPhrase, T extends BsRequest<R>, M extends ResponsePhraseWithId>
    IdentityHashMap<T, BsResponse<R, M>> getAuctionResultsBase(
            List<T> auctionRequests,
            Function<List<T>, List<ParsableRequest<IdentityHashMap<R, M>>>> parsableBsRequestsGetter
    ) {
        logger.trace("getAuctionResults {}", auctionRequests);
        int totalPhrases = auctionRequests.stream().map(BsRequest::getPhrases).mapToInt(Collection::size).sum();
        try (TraceProfile profile = Trace.current().profile("bsrank:bsrank_call", "", totalPhrases)) {
            List<ParsableRequest<IdentityHashMap<R, M>>> requests =
                    parsableBsRequestsGetter.apply(auctionRequests);

            ParallelFetcher<IdentityHashMap<R, M>> parallelFetcher =
                    fetcherFactory.getParallelFetcherWithSoftTimeout(softTimeoutSupplier.get());
            Map<Long, Result<IdentityHashMap<R, M>>> bsResults = parallelFetcher.execute(requests);
            IdentityHashMap<T, BsResponse<R, M>> result = new IdentityHashMap<>();
            for (int i = 0; i < auctionRequests.size(); i++) {
                T request = auctionRequests.get(i);
                BsResponse<R, M> bsResponse = convertResult(bsResults.get((long) i));
                result.put(request, bsResponse);
            }

            return result;
        } catch (InterruptedException ex) {
            logger.info("Interrupted while making requests {}", auctionRequests);
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    /**
     * Описания позиций-ключей в {@link #PLACE_TRAFARET_MAPPING}
     */
    public enum Place {
        PREMIUM1,
        PREMIUM2,
        PREMIUM3,
        PREMIUM4,
        GUARANTEE1,
        GUARANTEE2,
        GUARANTEE3,
        GUARANTEE4
    }

    public <R extends BsRequestPhrase, T extends BsRequest<R>>
    IdentityHashMap<T, BsResponse<R, PositionalBsTrafaretResponsePhrase>> getAuctionResults(List<T> auctionRequests) {
        if (auctionRequests.isEmpty()) {
            return new IdentityHashMap<>();
        }
        return getAuctionResultsBase(auctionRequests, t -> getParsableBsTrafaretRequests(t,
                PositionalBsTrafaretParser::new));
    }

    public <R extends BsRequestPhrase, T extends BsRequest<R>>
    IdentityHashMap<T, BsResponse<R, FullBsTrafaretResponsePhrase>> getAuctionResultsWithPositionCtrCorrection(
            List<T> auctionRequests) {
        if (auctionRequests.isEmpty()) {
            return new IdentityHashMap<>();
        }

        return getAuctionResultsBase(auctionRequests, t -> getParsableBsTrafaretRequests(t, FullBsTrafaretParser::new));
    }

    private <R extends BsRequestPhrase, T extends BsRequest<R>, M extends ResponsePhraseWithId>
    List<ParsableRequest<IdentityHashMap<R, M>>> getParsableBsTrafaretRequests(
            List<T> auctionRequests,
            Function<MicroMoneyConverter, BsTrafaretParser<M>> parserGetter) {
        logger.debug("Compose ParsableRequests with \"trafaret-auction\"=1");
        List<ParsableRequest<IdentityHashMap<R, M>>> requests = new ArrayList<>(auctionRequests.size());
        for (int i = 0; i < auctionRequests.size(); i++) {
            BsRequest<R> auctionRequest = auctionRequests.get(i);
            FastUrlBuilder builder = auctionRequest.getUrlBuilder(bsUrl);
            builder.addParam("trafaret-auction", 1);
            Request request = new RequestBuilder("GET")
                    .setUrl(builder.build())
                    .addHeader(HOST_HEADER, bsHostHeader)
                    .build();

            CurrencyCode currencyCode = auctionRequest.getCurrency().getCode();
            MicroMoneyConverter moneyConverter = MicroMoneyConverter.roundingUp(currencyCode);
            BsTrafaretParser<M> parser = parserGetter.apply(moneyConverter);

            ParsableBsTrafaretRequest<R, M> bsRequest =
                    new ParsableBsTrafaretRequest<>((long) i, request, auctionRequest, parser);
            requests.add(bsRequest);
        }
        return requests;
    }
}
