package ru.yandex.direct.bsauction.parser;

import java.util.List;
import java.util.function.Function;

import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsauction.BsAuctionBidItem;
import ru.yandex.direct.bsauction.FullBsTrafaretResponsePhrase;
import ru.yandex.direct.currency.Money;
import ru.yandex.yabs.server.proto.rank.TTrafaretClickometer;
import ru.yandex.yabs.server.proto.rank.TTrafaretClickometerRow;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class FullBsTrafaretParser implements BsTrafaretParser<FullBsTrafaretResponsePhrase> {
    private static final Logger logger = LoggerFactory.getLogger(FullBsTrafaretParser.class);

    private final Function<Long, Money> convertMicrosFunction;

    /**
     * @param convertMicrosFunction функция преобразования микроденег в {@link Money}
     */
    public FullBsTrafaretParser(Function<Long, Money> convertMicrosFunction) {
        this.convertMicrosFunction = convertMicrosFunction;
    }

    @Override
    public FullBsTrafaretResponsePhrase convertClickometer(TTrafaretClickometer clickometer) {
        logger.debug("Convert bsrank response with positionCtrCorrection for phrase {}", clickometer);

        boolean guaranteeNormalizationRequired = StreamEx.of(clickometer.getClickometerList())
                .filter(row -> row.getTrafaretID() == GUARANTEE_TRAFARET_ID)
                .filter(row -> row.getPosition() == 0)
                .findAny(row -> row.getX() == 1_000_000L)
                .isPresent();

        List<BsAuctionBidItem> bidItems =
                mapList(clickometer.getClickometerList(), convertClickometerRow(guaranteeNormalizationRequired));

        return new FullBsTrafaretResponsePhrase()
                .withId(Ints.checkedCast(clickometer.getTargetID()))
                .withBidItems(bidItems);
    }

    /**
     * Если {@code guaranteeNormalizationRequired} равен {@code true}, то уменьшаем значение
     * {@code positionCtrCorrection} для Гарантии, чтобы обеспечить монотонность его зависимости от ставки.
     */
    private Function<TTrafaretClickometerRow, BsAuctionBidItem> convertClickometerRow(
            boolean guaranteeNormalizationRequired) {
        return row -> {
            long positionCtrCorrection;
            if (guaranteeNormalizationRequired && row.getTrafaretID() == GUARANTEE_TRAFARET_ID) {
                // Сейчас отдаём для гарантии значения, уменьшенные в 10 раз. См. DIRECT-77521: Добавить корректировку на TrafficVolume для гарантии
                positionCtrCorrection = row.getX() / 10;
            } else {
                positionCtrCorrection = row.getX();
            }
            return new BsAuctionBidItem(positionCtrCorrection,
                    convertMicrosFunction.apply(row.getBid()), convertMicrosFunction.apply(row.getCpc()));
        };
    }
}
