package ru.yandex.direct.bsauction.parser;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.function.Function;

import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.commons.lang3.StringUtils;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.AbstractParsableRequest;
import ru.yandex.direct.bsauction.BsRequest;
import ru.yandex.direct.bsauction.BsRequestPhrase;
import ru.yandex.direct.bsauction.ResponsePhraseWithId;
import ru.yandex.direct.bsauction.utils.TrafaretAnswerParsingUtils;
import ru.yandex.yabs.server.proto.rank.TTrafaretClickometer;
import ru.yandex.yabs.server.proto.rank.TTrafaretRankAnswer;

import static org.apache.commons.lang3.StringUtils.LF;

/**
 * Экземпляр класса предназначен для разбора ответа от Торгов.
 */
public class ParsableBsTrafaretRequest<R extends BsRequestPhrase, M extends ResponsePhraseWithId> extends AbstractParsableRequest<IdentityHashMap<R, M>> {
    private static final Logger logger = LoggerFactory.getLogger(ParsableBsTrafaretRequest.class);

    private final BsRequest<R> bsRequest;
    private final BsTrafaretParser<M> bsTrafaretParser;

    public ParsableBsTrafaretRequest(long id, Request request, BsRequest<R> bsRequest,
                                     BsTrafaretParser<M> bsTrafaretParser) {
        super(id, request);
        this.bsRequest = bsRequest;
        this.bsTrafaretParser = bsTrafaretParser;
    }

    @Override
    public Function<Response, IdentityHashMap<R, M>> getParseFunction() {
        return this::parseBsResponse;
    }

    private IdentityHashMap<R, M> parseBsResponse(Response response) {
        String responseBody = response.getResponseBody(Charset.defaultCharset());
        return parseWholeResponseBody(responseBody);
    }

    private IdentityHashMap<R, M> parseWholeResponseBody(String responseBody) {
        String traferetAnswer = extractTrafaretAnswer(responseBody);
        return parseTrafaretAnswer(traferetAnswer);
    }

    /**
     * В ответе может приходить отладочная информация. Полезная нагрузка только в первой строке. Лишнее отрезаем.
     */
    private String extractTrafaretAnswer(String data) {
        return StringUtils.substringBefore(data, LF);
    }

    private IdentityHashMap<R, M> parseTrafaretAnswer(String traferetAnswer) {
        TTrafaretRankAnswer rankAnswer;
        try {
            rankAnswer = TrafaretAnswerParsingUtils.parseJson(traferetAnswer);
        } catch (InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("Can't parse given json bsrank answer");
        }

        Collection<M> responses = new ArrayList<>();
        Collection<Long> targetIdsWithoutClickometer = null;
        for (TTrafaretClickometer clickometer : rankAnswer.getTargetsList()) {
            // DIRECT-76125: Workaround в java для пустого второго ответа в трафаретных Торгах
            if (clickometer.getClickometerList().isEmpty()) {
                if (targetIdsWithoutClickometer == null) {
                    targetIdsWithoutClickometer = new ArrayList<>();
                }
                targetIdsWithoutClickometer.add(clickometer.getTargetID());
                continue;
            }
            responses.add(bsTrafaretParser.convertClickometer(clickometer));
        }
        if (targetIdsWithoutClickometer != null) {
            // В ответе были элементы без кликометра. Это нарушение протокола общения с ручкой Торгов.
            // Это не обязательно означает, что данных по этим фразам не было вообще.
            if (!targetIdsWithoutClickometer.isEmpty()) {
                logger.warn("There were missed clickometer for next targetIds: {}, bsrank url: {}",
                        targetIdsWithoutClickometer, bsRequest.getUrlBuilder("<bsrank.url>/").build());
            }
        }

        IdentityHashMap<R, M> requestResponseMap = new IdentityHashMap<>();
        List<R> requestPhrases = bsRequest.getPhrases();
        for (M responsePhrase : responses) {
            R requestPhrase = requestPhrases.get(responsePhrase.getId());

            requestResponseMap.put(requestPhrase, responsePhrase);
        }

        return requestResponseMap;
    }
}
