package ru.yandex.direct.bsauction.parser;

import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsauction.BsCpcPrice;
import ru.yandex.direct.bsauction.PositionalBsTrafaretResponsePhrase;
import ru.yandex.direct.currency.Money;
import ru.yandex.yabs.server.proto.rank.TTrafaretClickometer;
import ru.yandex.yabs.server.proto.rank.TTrafaretClickometerRow;

import static com.google.common.base.Preconditions.checkArgument;

public class PositionalBsTrafaretParser implements BsTrafaretParser<PositionalBsTrafaretResponsePhrase> {
    private static final Logger logger = LoggerFactory.getLogger(PositionalBsTrafaretParser.class);

    private final Function<Long, Money> convertMicrosFunction;

    /**
     * @param convertMicrosFunction функция преобразования микроденег в {@link Money}
     */
    public PositionalBsTrafaretParser(Function<Long, Money> convertMicrosFunction) {
        this.convertMicrosFunction = convertMicrosFunction;
    }

    /**
     * По описанию позиций в различных трафаретах для одной фразы ({@link TTrafaretClickometer})
     * заполняет {@link BsResponsePhrase}.
     * <p>
     * <b>Внимание:</b> в трафаретных Торгах приходит только информация о ставках в рекламном блоке.
     */
    @Override
    public PositionalBsTrafaretResponsePhrase convertClickometer(TTrafaretClickometer clickometer) {
        logger.debug("Convert bsrank response for phrase {}", clickometer);
        // todo maxlog: было бы здорово проверить, что в clickometer нет пропусков в данных
        /*
         Гарантия сейчас определяется трафаретом с ID=-1.
         С ней всё просто:
         1. выбираем все записи с этим trafaretId
         2. упорядочиваем записи по position
         3. конвертируем в BsCpcPrice
         */
        List<BsCpcPrice> guarantee = StreamEx.of(clickometer.getClickometerList())
                .filter(row -> row.getTrafaretID() == GUARANTEE_TRAFARET_ID)
                .sortedBy(TTrafaretClickometerRow::getPosition)
                .map(this::convertClickometerItem)
                .toList();

        /*
         Со спецразмещением (aka Premium) чуть сложнее. Описание одной позиции может прийти в нескольких трафаретах.
         Из описаний одной и той же позиции выбираем с наименьшей ценой. Если рекламодатель укажет эту цену,
         то попадёт на желаемую позицию.
         */
        BiPredicate<TTrafaretClickometerRow, TTrafaretClickometerRow>
                equalByPosition =
                (r1, r2) -> r1.getPosition() == r2.getPosition();
        BinaryOperator<TTrafaretClickometerRow> choseCheapest = (r1, r2) -> {
            if (r1.getBid() < r2.getBid()) {
                return r1;
            } else {
                return r2;
            }
        };

        List<BsCpcPrice> premium = StreamEx.of(clickometer.getClickometerList())
                .filter(row -> row.getTrafaretID() != GUARANTEE_TRAFARET_ID)
                .sortedBy(TTrafaretClickometerRow::getPosition)
                .collapse(equalByPosition, choseCheapest)
                .map(this::convertClickometerItem)
                .toList();

        checkArgument(!premium.isEmpty(), "No data for premium block");
        checkArgument(!guarantee.isEmpty(), "No data for guarantee block");

        // Если вдруг пришло меньше 4-ёх позиций, заполняем нехватку последней позицией
        while (premium.size() < 4) {
            premium.add(premium.get(premium.size() - 1));
        }

        PositionalBsTrafaretResponsePhrase result = new PositionalBsTrafaretResponsePhrase();
        result.setId(Ints.checkedCast(clickometer.getTargetID()));
        result.setPremium(premium.toArray(new BsCpcPrice[0]));
        result.setGuarantee(guarantee.toArray(new BsCpcPrice[0]));
        return result;
    }

    /**
     * Преобразует {@link TTrafaretClickometerRow} в {@link BsCpcPrice}
     */
    BsCpcPrice convertClickometerItem(TTrafaretClickometerRow row) {
        return new BsCpcPrice(convertMicrosFunction.apply(row.getCpc()), convertMicrosFunction.apply(row.getBid()));
    }
}
