package ru.yandex.direct.bshistory;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;

@ParametersAreNonnullByDefault
public class History {
    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(History.class);

    private static final String ORDER_PREFIX = "O";
    private static final String AD_GROUP_PREFIX = "G";
    private static final String PHRASE_PREFIX = "P";
    private static final String IMAGE_BANNER_PREFIX = "im";

    private @Nullable
    Long orderId;
    private @Nullable
    Long adGroupId;
    private final List<BigInteger> phraseBsIds;
    private final Map<Long, List<Long>> bannerIdToBannerBsIds;
    private final Map<Long, List<Long>> imageBannerIdToBannerBsIds;

    public History(@Nullable Long orderId, @Nullable Long adGroupId,
                   List<BigInteger> phraseBsIds, Map<Long, List<Long>> bannerIdToBannerBsIds,
                   Map<Long, List<Long>> imageBannerIdToBannerBsIds) {
        this.orderId = orderId;
        this.adGroupId = adGroupId;
        this.phraseBsIds = unmodifiableList(phraseBsIds);
        this.bannerIdToBannerBsIds = unmodifiableMapOfLists(bannerIdToBannerBsIds);
        this.imageBannerIdToBannerBsIds = unmodifiableMapOfLists(imageBannerIdToBannerBsIds);
    }

    public History(@Nullable Long orderId, @Nullable Long adGroupId,
                   BigInteger phraseBsId, Map<Long, Long> bannerIdToBannerBsIds,
                   Map<Long, Long> imageBannerIdToBannerBsIds) {
        this(orderId, adGroupId, singletonList(phraseBsId), wrapMapValuesInLists(bannerIdToBannerBsIds),
                wrapMapValuesInLists(imageBannerIdToBannerBsIds));
    }

    private static <K, V> Map<K, List<V>> wrapMapValuesInLists(Map<K, V> map) {
        return EntryStream.of(map)
                .mapValues(Collections::singletonList)
                .toMap();
    }

    private <K, V> Map<K, List<V>> unmodifiableMapOfLists(Map<K, List<V>> m) {
        return EntryStream.of(m)
                .mapValues(Collections::unmodifiableList)
                .toMap();
    }

    /**
     * Парсит строку c историей которая хранится в таблице bids_phraseid_history.phraseIdHistory
     *
     * @param serializedHistory строка вида <pre>O<orderId>;G<adGroupId>;P<phraseBsId>,...;<bannerId>:<bannerBsId>,...</pre>
     * @return объект {@link History}
     */
    public static History parse(String serializedHistory) {
        Long orderId = null;
        Long adGroupId = null;
        List<BigInteger> phraseBsIds = new ArrayList<>();
        Map<Long, List<Long>> bannerIdToBannerBsIds = new HashMap<>();
        Map<Long, List<Long>> imageBannerIdToBannerBsIds = new HashMap<>();
        List<String> fields = StreamEx.split(serializedHistory, ';').toList();
        for (String field : fields) {
            if (field.startsWith(ORDER_PREFIX)) {
                orderId = Long.parseLong(field.substring(ORDER_PREFIX.length()));
            } else if (field.startsWith(AD_GROUP_PREFIX)) {
                adGroupId = Long.parseLong(field.substring(AD_GROUP_PREFIX.length()));
            } else if (field.startsWith(PHRASE_PREFIX)) {
                phraseBsIds = StreamEx.split(field.substring(PHRASE_PREFIX.length()), ',')
                        .map(phraseBsId -> phraseBsId.equals("") ? null : new BigInteger(phraseBsId))
                        .nonNull()
                        .toList();
            } else {
                List<String> ids = StreamEx.split(field, ':').toList();
                if (ids.size() == 2) {
                    List<Long> bannerBsIds = StreamEx.split(ids.get(1), ',').map(Long::parseLong).toList();
                    if (field.startsWith(IMAGE_BANNER_PREFIX)) {
                        String bannerIdText = ids.get(0).substring(IMAGE_BANNER_PREFIX.length());
                        if (bannerIdText.isEmpty()) {
                            logger.warn("Missed imageBannerId in phraseIdHistory \"{}\"", serializedHistory);
                            continue;
                        }
                        Long bannerId = Long.parseLong(bannerIdText);
                        imageBannerIdToBannerBsIds.put(bannerId, bannerBsIds);
                    } else {
                        String bannerIdText = ids.get(0);
                        if (bannerIdText.isEmpty()) {
                            logger.warn("Missed bannerId in phraseIdHistory \"{}\"", serializedHistory);
                            continue;
                        }
                        Long bannerId = Long.parseLong(bannerIdText);
                        bannerIdToBannerBsIds.put(bannerId, bannerBsIds);
                    }

                }
            }

        }
        return new History(orderId, adGroupId, phraseBsIds, bannerIdToBannerBsIds, imageBannerIdToBannerBsIds);
    }

    @Nullable
    public Long getOrderId() {
        return orderId;
    }

    @Nullable
    public Long getAdGroupId() {
        return adGroupId;
    }

    public List<BigInteger> getPhraseBsIds() {
        return phraseBsIds;
    }

    public Map<Long, List<Long>> getBannerIdToBannerBsIds() {
        return bannerIdToBannerBsIds;
    }

    public List<Long> getBannerBsIdsByBannerId(Long bannerId) {
        return bannerIdToBannerBsIds.get(bannerId);
    }

    public Map<Long, List<Long>> getImageBannerIdToBannerBsIds() {
        return imageBannerIdToBannerBsIds;
    }

    public boolean isEmpty() {
        return orderId == null && adGroupId == null && phraseBsIds.isEmpty() && bannerIdToBannerBsIds.isEmpty()
                && imageBannerIdToBannerBsIds.isEmpty();
    }

    /**
     * Сериализует объект {@link History} в строку c историей которая хранится в таблице bids_phraseid_history.phraseIdHistory
     *
     * @return строка вида <pre>O<orderId>;G<adGroupId>;P<phraseBsId>,...;<bannerId>:<bannerBsId>,...</pre>
     */
    public String serialize() {
        List<String> fields = new ArrayList<>();
        if (this.orderId != null) {
            fields.add(ORDER_PREFIX + this.orderId);
        }
        if (this.adGroupId != null) {
            fields.add(AD_GROUP_PREFIX + this.adGroupId);
        }
        if (!this.phraseBsIds.isEmpty()) {
            fields.add(PHRASE_PREFIX + String.join(",", this.phraseBsIds.stream()
                    .map(BigInteger::toString).collect(Collectors.toList()))
            );
        }
        if (!this.imageBannerIdToBannerBsIds.isEmpty()) {
            this.imageBannerIdToBannerBsIds.forEach((bannerId, bannerBsIds) ->
                    fields.add(IMAGE_BANNER_PREFIX + bannerId + ":" + StreamEx.of(bannerBsIds)
                            .map(bannerBsId -> bannerBsId == null ? "" : bannerBsId)
                            .joining(","))
            );
        }
        if (!this.bannerIdToBannerBsIds.isEmpty()) {
            this.bannerIdToBannerBsIds.forEach((bannerId, bannerBsIds) ->
                    fields.add(bannerId + ":" + StreamEx.of(bannerBsIds)
                            .map(bannerBsId -> bannerBsId == null ? "" : bannerBsId)
                            .joining(","))
            );
        }
        return String.join(";", fields);
    }

    @Override
    public String toString() {
        return "ru.yandex.direct.bshistory.History{" +
                "orderId=" + orderId +
                ", adGroupId=" + adGroupId +
                ", phraseBsIds=" + phraseBsIds +
                ", bannerIdToBannerBsIds=" + bannerIdToBannerBsIds +
                ", imageBannerIdToBannerBsIds=" + imageBannerIdToBannerBsIds +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        History that = (History) o;
        return Objects.equals(serialize(), that.serialize());
    }

    @Override
    public int hashCode() {
        return Objects.hash(serialize());
    }
}
