package ru.yandex.direct.bshistory;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

@ParametersAreNonnullByDefault
public final class HistoryUtils {

    private HistoryUtils() {
    }

    /**
     * Извлекает из истории {@link History} историю по баннеру в виде строки.
     *
     * @return bannerSerializedHistory строка вида <pre>(B<bannerBsId>,phraseBsId),(...),...</pre>
     */
    @Nullable
    public static String convertHistoryForBanner(History history) {
        if (history.getBannerIdToBannerBsIds().isEmpty() || history.getPhraseBsIds().isEmpty()) {
            return null;
        }
        List<Long> bannerBsIds = history.getBannerIdToBannerBsIds().values()
                .stream().flatMap(Collection::stream).collect(Collectors.toList());

        List<String> bannerHistoryItems = new ArrayList<>();
        for (Long bannerBsId : bannerBsIds) {
            bannerHistoryItems.addAll(
                    history.getPhraseBsIds().stream()
                            .map(id -> "B" + bannerBsId + "," + id)
                            .collect(Collectors.toList())
            );
        }
        return String.join(",", bannerHistoryItems);
    }

    /**
     * Извлекает из истории {@link History} историю по группе в виде строки
     *
     * @return adGroupSerializedHistory строка вида <pre>(G<adGroupId>,phraseBsId),(...),...</pre>
     */
    @Nullable
    public static String convertHistoryForAdGroup(History history, long adGroupId) {
        if (history.isEmpty()) {
            return null;
        }
        List<Long> adGroupIds = StreamEx.of(history.getAdGroupId(), adGroupId).nonNull().toList();

        List<String> adGroupHistoryItems = new ArrayList<>();
        for (Long groupId : adGroupIds) {
            adGroupHistoryItems.addAll(
                    history.getPhraseBsIds().stream()
                            .map(id -> "G" + groupId + "," + id)
                            .collect(Collectors.toList())
            );
        }
        return String.join(",", adGroupHistoryItems);
    }

    /**
     * Добавляет к строке с историей по баннеру строку с записью о еще одном баннере - B<bannerBsId>,phraseBsId
     *
     * @param bannerSerializedHistory строка вида <pre>(B<bannerBsId>,phraseBsId),(...),...</pre>
     * @param bannerBsId              id баннера в БК
     * @param phraseBsId              id ключевой фразы в БК
     * @return bannerSerializedHistory
     */
    public static String prependHistoryForBanner(@Nullable String bannerSerializedHistory, long bannerBsId,
                                                 BigInteger phraseBsId) {
        //BS::History::prepend_history
        String history = "B" + bannerBsId + "," + phraseBsId;
        return StreamEx.of(history, bannerSerializedHistory).filter(StringUtils::isNotBlank).joining(",");
    }

    /**
     * Добавляет к строке с историей по группе строку с записью о еще одной группе - G<adGroupId>,phraseBsId
     *
     * @param adGroupSerializedHistory строка вида <pre>(G<adGroupId>,phraseBsId),(...),...</pre>
     * @param adGroupId                id группы
     * @param phraseBsId               id ключевой фразы в БК
     * @return adGroupSerializedHistory
     */
    public static String prependHistoryForAdGroup(@Nullable String adGroupSerializedHistory, long adGroupId,
                                                  BigInteger phraseBsId) {
        //BS::History::prepend_history
        String history = "G" + adGroupId + "," + phraseBsId;
        return StreamEx.of(history, adGroupSerializedHistory).filter(StringUtils::isNotBlank).joining(",");
    }


}

