package ru.yandex.direct.bsexport.messaging;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import com.google.common.base.CharMatcher;
import com.google.common.collect.Iterables;
import com.google.protobuf.Descriptors;
import com.google.protobuf.MapEntry;
import com.google.protobuf.Message;
import com.google.protobuf.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsexport.model.FeedRequestMessage;
import ru.yandex.direct.bsexport.model.Order;
import ru.yandex.direct.bsexport.model.UpdateData2Request;

import static com.google.protobuf.TextFormat.unsignedToString;

/**
 * Сериализация protobuf в {@code perl SOAP::Lite}-подобное SOAP-сообщение.
 * <p>
 * Принцип работы с proto-сообщением творчески скопирован с {@code com.googlecode.protobuf.format.XmlFormat}
 * из {@code contrib/java/com/googlecode/protobuf-java-format/protobuf-java-format}
 * <p>
 *
 * @implNote Особенности серализации (отличия от perl):
 * <pre>><ul>
 *  <li>структура с данными запроса переименована в {@literal request}, так как в perl названия не имеет
 *      (подставляется автоматически сгенерированное {@literal c-gensym3}</li>
 *   <li>номер воркера (второй позиционный параметр) назван {@literal workerID}, в оргинале имел автоматически
 *       сгенерированное {@literal c-gensym5}</li>
 *  <li>у поля {@code QueueSetTime} изменен тип на строковый. в фиде ездит строками, в bssoap читается как строки</li>
 *  <li>изменен порядок полей внутри объектов с типом {@literal namesp2:SOAPStruct}</li>
 *  <li>не переиспользуются узлы документа (через ссылки), всегда значения сериализованы in-place</li>
 *  <li>в строках всегда эксейпится {@code >}</li>
 * </ul></pre
 */
public class SoapSerializer {
    private static final Logger logger = LoggerFactory.getLogger(SoapSerializer.class);

    private static final Base64.Encoder BASE64 = Base64.getEncoder();
    private static final Charset CHARSET = StandardCharsets.UTF_8;

    private static final byte[] START_DOCUMENT =
            String.format("<?xml version=\"1.0\" encoding=\"%s\"?>", CHARSET).getBytes();

    private static final String W3C_XML_SCHEMA_NS_URI = "http://www.w3.org/1999/XMLSchema";
    private static final String W3C_XML_SCHEMA_INSTANCE_NS_URI = "http://www.w3.org/1999/XMLSchema-instance";
    private static final String W3C_SOAP_ENCODING_NS_URI = "http://schemas.xmlsoap.org/soap/encoding/";
    private static final String NS1_PREFIX = "namesp1";
    private static final String NS2_PREFIX = "namesp2";
    private static final String SCHEMA_PREFIX = "xsd";
    private static final String SCHEMA_INSTANCE_PREFIX = "xsi";
    private static final String SOAP_ENCODING_PREFIX = "SOAP-ENC";
    private static final String TYPE_INT = SCHEMA_PREFIX + ':' + "int";
    private static final String TYPE_STRING = SCHEMA_PREFIX + ':' + "string";
    private static final String TYPE_ANY = SCHEMA_PREFIX + ':' + "ur-type";
    private static final String TYPE_STRUCT = NS2_PREFIX + ':' + "SOAPStruct";
    private static final String TYPE_BASE64 = SOAP_ENCODING_PREFIX + ':' + "base64";
    private static final String TYPE_ARRAY = SOAP_ENCODING_PREFIX + ':' + "Array";

    private static final Set<Descriptors.FieldDescriptor> MAPS_AS_STRUCT_FIELDS = Set.of(
            UpdateData2Request.getDescriptor().findFieldByName(FieldNames.ORDER),
            Order.getDescriptor().findFieldByName(FieldNames.CONTEXT)
    );

    private static final Map<Descriptors.FieldDescriptor, String> SOAP_TYPE_OVERRIDES = Map.of(
    );

    private static final Set<Descriptors.FieldDescriptor> NO_SOAP_FIELDS = Set.of(
            FeedRequestMessage.getDescriptor().findFieldByName("data"),
            Order.getDescriptor().findFieldByName(FieldNames.TARGETING_EXPRESSION_V2),
// TODO: окнуть и раскомментировать
//            Order.getDescriptor().findFieldByName("DirectDeals"),
            Order.getDescriptor().findFieldByName("AllowAloneTrafaret"),
            Order.getDescriptor().findFieldByName("ImpressionStandardTime"),
            Order.getDescriptor().findFieldByName("ImpressionStandardType"),
            Order.getDescriptor().findFieldByName("HasTurboApp"),
            Order.getDescriptor().findFieldByName("RequireFiltrationByDontShowDomains"),
            Order.getDescriptor().findFieldByName("OrderCreateTime"),
            Order.getDescriptor().findFieldByName("EshowsBannerRate"),
            Order.getDescriptor().findFieldByName("EshowsVideoRate"),
            Order.getDescriptor().findFieldByName("EshowsVideoType"),
            Order.getDescriptor().findFieldByName("BillingOrders")
    );

    public static final String SOAP_URI = "YaBSSOAPExport";
    public static final String SOAP_METHOD = "UpdateData2";

    // todo - выселить это в отдельный от сериализации класс
    private final SOAPMessage soapMessage;

    private final SOAPEnvelope envelope;
    private final QName xsiType;
    private final QName arrayType;
    private final QName xsiNull;

    private int fieldsCount;

    public SoapSerializer() throws SOAPException {
        MessageFactory messageFactory = MessageFactory.newInstance();
        soapMessage = messageFactory.createMessage();
        envelope = soapMessage.getSOAPPart().getEnvelope();
        removeHeader();

        configureEnvelope();
        // работает только после задания namespace в envelope
        xsiType = envelope.createQName("type", SCHEMA_INSTANCE_PREFIX);
        xsiNull = envelope.createQName("null", SCHEMA_INSTANCE_PREFIX);
        arrayType = envelope.createQName("arrayType", SOAP_ENCODING_PREFIX);

        fieldsCount = 0;
    }

    /**
     * Удаляем {@literal <SOAP-ENV:Header/>}, в SOAP::Lite его не было
     */
    private void removeHeader() throws SOAPException {
        envelope.getHeader().detachNode();
    }

    /**
     * Задаем атрибуты в {@literal <SOAP-ENV:Envelope} как в SOAP::Lite
     */
    private void configureEnvelope() throws SOAPException {
        envelope.setEncodingStyle(W3C_SOAP_ENCODING_NS_URI);
        envelope.addNamespaceDeclaration(SOAP_ENCODING_PREFIX, W3C_SOAP_ENCODING_NS_URI);
        envelope.addNamespaceDeclaration(NS2_PREFIX, "http://xml.apache.org/xml-soap");
        envelope.addNamespaceDeclaration(SCHEMA_PREFIX, W3C_XML_SCHEMA_NS_URI);
        envelope.addNamespaceDeclaration(SCHEMA_INSTANCE_PREFIX, W3C_XML_SCHEMA_INSTANCE_NS_URI);
    }

    public String serializeRoot(Message message) throws SOAPException, IOException {
        SOAPBody soapBody = envelope.getBody();

        SOAPBodyElement updateData2 = soapBody.addBodyElement(envelope.createName(SOAP_METHOD, NS1_PREFIX, SOAP_URI));
        print(message, updateData2);

        // выделяем буфер исходя из средней оценки длины поля
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream(40 * fieldsCount);
        byteArrayOutputStream.writeBytes(START_DOCUMENT);
        soapMessage.writeTo(byteArrayOutputStream);

        logger.debug("serialized message with {} fields to {} bytes", fieldsCount, byteArrayOutputStream.size());
        return byteArrayOutputStream.toString();
    }


    private void print(Message message, SOAPElement soapElement) throws SOAPException {
        for (Map.Entry<Descriptors.FieldDescriptor, Object> field : message.getAllFields().entrySet()) {
            printField(field.getKey(), field.getValue(), soapElement);
        }
    }

    private void printField(Descriptors.FieldDescriptor field, Object value, SOAPElement soapElement)
            throws SOAPException {
        // TODO поддержка Map
        if (field.isMapField()) {
            //noinspection unchecked
            List<MapEntry<?, ?>> mapEntries = (List<MapEntry<?, ?>>) value;
            if (MAPS_AS_STRUCT_FIELDS.contains(field)) {
                printMapAsStruct(field, mapEntries, soapElement);
            } else {
                throw new UnsupportedOperationException("maps is not supported");
            }

        } else if (field.isRepeated()) {
            List<?> list = (List<?>) value;
            if (list.isEmpty()) {
                // TODO поддержка пустых массивов
                throw new UnsupportedOperationException("empty repeatable is not supported");
            }
            printRepeatedField(field, list, soapElement);
        } else {
            printSingleField(field, value, soapElement);
        }
    }

    private void printRepeatedField(Descriptors.FieldDescriptor field, List<?> list, SOAPElement soapElement)
            throws SOAPException {
        SOAPElement childElement = soapElement.addChildElement(field.getName());

        Set<String> elementTypes = new HashSet<>();
        for (Object element : list) {
            String type = printRepeatedField(field, element, childElement);
            elementTypes.add(type);
        }
        String elementType = elementTypes.size() == 1 ? Iterables.getOnlyElement(elementTypes) : TYPE_ANY;
        if (elementType == null) {
            // в SOAP сериализацется так
            elementType = TYPE_ANY;
        }
        childElement.addAttribute(arrayType, elementType + '[' + list.size() + ']');
        childElement.addAttribute(xsiType, TYPE_ARRAY);
    }

    private void printMapAsStruct(Descriptors.FieldDescriptor field, List<MapEntry<?, ?>> value,
                                  SOAPElement soapElement)
            throws SOAPException {
        SOAPElement childElement = soapElement.addChildElement(field.getName());
        addTypeAttribute(childElement, TYPE_STRUCT);

        for (MapEntry<?, ?> element : value) {
            Descriptors.FieldDescriptor keyField = element.getDescriptorForType().findFieldByName("key");
            Descriptors.FieldDescriptor valueField = element.getDescriptorForType().findFieldByName("value");

            String mapKey;
            switch (keyField.getType()) {
                case STRING:
                    mapKey = (String) element.getKey();
                    break;

                default:
                    String error = String.format("Type %s of map key '%s' (in map %s) is not supported",
                            keyField.getType(), element.getKey(), field.getName());
                    throw new UnsupportedOperationException(error);
            }

            SOAPElement structElement = childElement.addChildElement(mapKey);
            String type = printFieldValue(valueField, element.getValue(), structElement);
            addTypeAttribute(structElement, type);
        }
    }

    /**
     * Добавить аттрибут, отвечающий за тип данных
     *
     * @param soapElement SOAP-элемент, тип которого задаем
     * @param type        строка со значением типа или {@code null}
     */
    private void addTypeAttribute(SOAPElement soapElement, String type) throws SOAPException {
        if (type == null) {
            soapElement.addAttribute(xsiNull, "1");
        } else {
            soapElement.addAttribute(xsiType, type);
        }
    }

    private void checkField(Descriptors.FieldDescriptor field) {
        if (field.isExtension()) {
            throw new UnsupportedOperationException("extensions are not supported");
        }
        if (field.getType() == Descriptors.FieldDescriptor.Type.GROUP) {
            throw new UnsupportedOperationException("groups are not supported");
        }
    }

    private void printSingleField(Descriptors.FieldDescriptor field, Object value, SOAPElement soapElement)
            throws SOAPException {
        checkField(field);

        SOAPElement childElement = soapElement.addChildElement(field.getName());
        String type = printFieldValue(field, value, childElement);
        addTypeAttribute(childElement, type);
    }

    /**
     * Добавить элемент массива
     *
     * @param field       спецификация поля (массива)
     * @param value       значение для добавления
     * @param soapElement SOAP-элемент (массив), в который добавляем элементы
     * @return SOAP-тип значения value
     */
    private String printRepeatedField(Descriptors.FieldDescriptor field, Object value, SOAPElement soapElement)
            throws SOAPException {
        checkField(field);

        SOAPElement childElement = soapElement.addChildElement("item");
        String type = printFieldValue(field, value, childElement);
        addTypeAttribute(childElement, type);
        return type;
    }

    /**
     * Задать значение SOAP-элемента
     *
     * @param field       спецификация поля
     * @param value       значение
     * @param soapElement SOAP-элемент, в котором задаем значение
     * @return SOAP-тип значения value
     */
    private String printFieldValue(Descriptors.FieldDescriptor field, Object value, SOAPElement soapElement)
            throws SOAPException {
        fieldsCount++;

        if (NO_SOAP_FIELDS.contains(field)) {
            soapElement.setValue("skipped in SOAP");
            return TYPE_STRING;
        }

        if (SOAP_TYPE_OVERRIDES.containsKey(field)) {
            return printValueWithOverridenType(field, value, soapElement);
        }

        switch (field.getType()) {
            case INT32:
            case INT64:
            case SINT32:
            case SINT64:
            case SFIXED32:
            case SFIXED64:
//            case FLOAT:
//            case DOUBLE:
//            case BOOL:
                // Good old toString() does what we want for these types.
                soapElement.setValue(value.toString());
                return TYPE_INT;

            case UINT32:
            case FIXED32:
                soapElement.setValue(unsignedToString((Integer) value));
                return TYPE_INT;

            case UINT64:
            case FIXED64:
                soapElement.setValue(unsignedToString((Long) value));
                return TYPE_INT;

            case STRING:
                String result = (String) value;

                if (encodeNeeded(result)) {
                    byte[] stringBytes = (result.getBytes(CHARSET));
                    soapElement.setValue(BASE64.encodeToString(stringBytes));
                    return TYPE_BASE64;
                } else {
                    soapElement.setValue(result);
                    return TYPE_STRING;
                }

//            case BYTES:
//                return null;

            case ENUM:
                soapElement.setValue((((Descriptors.EnumValueDescriptor) value).getName()));
                return TYPE_STRING;


            case MESSAGE:
                // Value types
                if (field.getMessageType().getFullName().equals(Value.getDescriptor().getFullName())) {
                    Value typedValue = (Value) value;
                    switch (typedValue.getKindCase()) {
                        case NULL_VALUE:
                            // особая обработка NULL-значений
                            return null;
                        default:
                            throw new UnsupportedOperationException("value serialization not implemented");
                    }
                }

                print((Message) value, soapElement);
                return TYPE_STRUCT;

            default:
                throw new UnsupportedOperationException("type " + field.getType() + " of field " + field.getName()
                        + " is not supported");

        }
    }

    private String printValueWithOverridenType(Descriptors.FieldDescriptor field, Object value, SOAPElement soapElement)
            throws SOAPException {
        String soapType = SOAP_TYPE_OVERRIDES.get(field);

        if (TYPE_INT.equals(soapType) && field.getType() == Descriptors.FieldDescriptor.Type.STRING) {
            String stringValue = (String) value;
            // пока считаем, что должно помещаться в Long
            try {
                Long.parseLong(stringValue);
            } catch (NumberFormatException e) {
                throw new SOAPException("Can't serialize field " + field.getName() + " as " + TYPE_INT, e);
            }
            soapElement.setValue(stringValue);
        } else {
            throw new UnsupportedOperationException("Overriding type " + field.getType()
                    + " of field " + field.getName() + " to " + soapType + " is not supported");
        }
        return soapType;
    }

    private static final CharMatcher STRING_MATCHER = CharMatcher.inRange(' ', '~')
            .or(CharMatcher.is('\t'))
            .or(CharMatcher.is('\n'))
            .or(CharMatcher.is('\r'))
            .negate();

    private static boolean encodeNeeded(String string) {
        return STRING_MATCHER.matchesAnyOf(string);
    }

}
