package ru.yandex.direct.bsexport.messaging.processors;

import java.util.Map;
import java.util.function.Consumer;

import org.w3c.dom.Node;

import ru.yandex.direct.bsexport.messaging.FieldNames;
import ru.yandex.direct.bsexport.model.ContextResponse;

class ContextProcessor extends Processor<ContextResponse.Builder> {

    private static final Map<String, FieldProcessor<ContextResponse.Builder, ?>> PROCESSORS = Map.of(
            FieldNames.EID, new LongFieldProcessor<>(ContextResponse.Builder::setEID),
            FieldNames.ID, new LongFieldProcessor<>(ContextResponse.Builder::setID),
            FieldNames.ERROR, new BooleanFieldProcessor<>(ContextResponse.Builder::setError),
            FieldNames.ERROR_MESSAGE, new StringFieldProcessor<>(ContextResponse.Builder::setErrorMessage)
    );

    ContextProcessor() {
        super(ContextResponse::newBuilder, PROCESSORS);
    }

    @Override
    Consumer<Node> getChildFieldsProcessor(String nodeName) {
        switch (nodeName) {
            case FieldNames.PHRASE:
                return this::processPhrase;
            case FieldNames.BANNER:
                return this::processBanner;
            case FieldNames.GOAL_CONTEXT:
                return this::processRetargeting;
            case FieldNames.DYNAMIC:
                return this::processDynamic;

            default:
                return Processor::doNothing;
        }
        // TODO: BANNERS, PHRASES, ....
    }

    ContextResponse getResult() {
        return builder.build();
    }

    private void processBanner(Node bannerNode) {
        var processor = new BannerProcessor();
        FieldProcessor.processEachChildNode(bannerNode, processor::processFields);
        builder.putBANNER(bannerNode.getNodeName(), processor.getResult());
    }

    private void processPhrase(Node phraseNode) {
        var processor = new PhraseProcessor();
        FieldProcessor.processEachChildNode(phraseNode, processor::processFields);
        builder.putPHRASE(phraseNode.getNodeName(), processor.getResult());
    }

    private void processRetargeting(Node goalContextNode) {
        var processor = new GoalContextProcessor();
        FieldProcessor.processEachChildNode(goalContextNode, processor::processFields);
        String key = processor.getKey();
        if (key == null) {
            key = goalContextNode.getNodeName();
        }
        builder.putGOALCONTEXT(key, processor.getResult());
    }

    private void processDynamic(Node dynamicNode) {
        var processor = new DynamicProcessor();
        FieldProcessor.processEachChildNode(dynamicNode, processor::processFields);
        String key = processor.getKey();
        if (key == null) {
            key = dynamicNode.getNodeName();
        }
        builder.putDYNAMIC(key, processor.getResult());
    }
}
