package ru.yandex.direct.canvas.client;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.canvas.client.model.CanvasRequest;
import ru.yandex.direct.canvas.client.model.exception.CanvasClientException;
import ru.yandex.direct.canvas.client.model.ffmpegresolutions.FfmpegResolutionsRequestType;
import ru.yandex.direct.canvas.client.model.ffmpegresolutions.FfmpegResolutionsResponse;
import ru.yandex.direct.canvas.client.model.ffmpegresolutions.GetFfmpegResolutionsRequest;
import ru.yandex.direct.canvas.client.model.html5.CreateHtml5BatchRequest;
import ru.yandex.direct.canvas.client.model.html5.Html5BatchResponse;
import ru.yandex.direct.canvas.client.model.html5.Html5SourceResponse;
import ru.yandex.direct.canvas.client.model.html5.Html5Tag;
import ru.yandex.direct.canvas.client.model.html5.Html5ValidationErrorResponse;
import ru.yandex.direct.canvas.client.model.html5.UploadHtml5CreativeToDirectRequest;
import ru.yandex.direct.canvas.client.model.html5.UploadHtml5Request;
import ru.yandex.direct.canvas.client.model.video.AdditionResponse;
import ru.yandex.direct.canvas.client.model.video.CreateDefaultAdditionRequest;
import ru.yandex.direct.canvas.client.model.video.CreateVideoFromFileRequest;
import ru.yandex.direct.canvas.client.model.video.CreateVideoFromUrlRequest;
import ru.yandex.direct.canvas.client.model.video.Creative;
import ru.yandex.direct.canvas.client.model.video.CreativeResponse;
import ru.yandex.direct.canvas.client.model.video.GenerateConditions;
import ru.yandex.direct.canvas.client.model.video.GenerateVideoAdditions;
import ru.yandex.direct.canvas.client.model.video.GetCreatedVideoRequest;
import ru.yandex.direct.canvas.client.model.video.GetPresetsRequest;
import ru.yandex.direct.canvas.client.model.video.GetUcVideoAdditionsRequest;
import ru.yandex.direct.canvas.client.model.video.GetVideoAdditionsRequest;
import ru.yandex.direct.canvas.client.model.video.GetVideoByIdRequest;
import ru.yandex.direct.canvas.client.model.video.GetVideoRequest;
import ru.yandex.direct.canvas.client.model.video.PresetResponse;
import ru.yandex.direct.canvas.client.model.video.UacVideoCreativeType;
import ru.yandex.direct.canvas.client.model.video.UcCreativeResponse;
import ru.yandex.direct.canvas.client.model.video.VideoResponse;
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse;
import ru.yandex.direct.canvas.client.model.video.VideoValidationErrorResponse;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static io.netty.handler.codec.http.HttpHeaderNames.ACCEPT_LANGUAGE;
import static io.netty.handler.codec.http.HttpHeaderNames.AUTHORIZATION;
import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static org.asynchttpclient.util.HttpConstants.Methods.GET;
import static org.asynchttpclient.util.HttpConstants.Methods.POST;
import static ru.yandex.direct.canvas.client.model.ffmpegresolutions.FfmpegResolutionsRequestType.OUTDOOR_TYPE;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.utils.CommonUtils.safeCast;

/**
 * Клиент для Canvas
 */
@ParametersAreNonnullByDefault
public class CanvasClient {
    private static final String PATH_DELIMITER = "/";
    private static final Logger logger = LoggerFactory.getLogger(CanvasClient.class);
    public static final String CANVAS_CLIENT = "canvas.client";

    private final CanvasClientConfiguration config;
    private final ParallelFetcherFactory parallelFetcherFactory;
    private final Supplier<List<FfmpegResolutionsResponse>> outdoorFfmpegResolutionsCache;

    public CanvasClient(CanvasClientConfiguration config, ParallelFetcherFactory parallelFetcherFactory) {
        this.config = config;
        this.parallelFetcherFactory = parallelFetcherFactory;
        this.outdoorFfmpegResolutionsCache = createOutdoorFfmpegResolutionsCache();
    }

    public CanvasClient(CanvasClientConfiguration config, AsyncHttpClient asyncHttpClient) {
        this(config, new ParallelFetcherFactory(asyncHttpClient,
                new FetcherSettings().withRequestTimeout(config.getReadTimeout())));
    }

    /**
     * По переданному массиву creativeIds получить video_addition-креативы из Canvas с кастомным timeout
     */
    public List<CreativeResponse> getVideoAdditions(Long clientId, List<Long> creativeIds, Duration timeout) {
        ArrayList<CreativeResponse> creativeResponses = StreamEx.ofSubLists(creativeIds, config.getChunkSize())
                .map(creativesSubList -> new GetVideoAdditionsRequest(clientId, creativesSubList))
                .map(request -> doRequest(request, config.getCanvasVideoUrl(), timeout, GET))
                .toFlatCollection(Function.identity(), ArrayList::new);
        return creativeResponses;
    }

    /**
     * По переданному массиву creativeIds получить video_addition-креативы для uc-кампаний из Canvas
     * с кастомным timeout и нужным форматом
     */
    public List<UcCreativeResponse> getUcCreativeData(Long clientId, List<Long> creativeIds, Duration timeout) {
        ArrayList<UcCreativeResponse> creativeResponses = StreamEx.ofSubLists(creativeIds, config.getChunkSize())
                .map(creativesSubList -> new GetUcVideoAdditionsRequest(clientId, creativesSubList))
                .map(request -> doRequest(request, config.getCanvasVideoUrl(), timeout, GET))
                .toFlatCollection(Function.identity(), ArrayList::new);
        return creativeResponses;
    }

    /**
     * По переданному массиву creativeIds получить video_addition-креативы из Canvas.
     */
    public List<CreativeResponse> getVideoAdditions(Long clientId, List<Long> creativeIds) {
        return getVideoAdditions(clientId, creativeIds, config.getReadTimeout());
    }

    /**
     * По переданному массиву creativeIds получить video_addition-креативы для uc-кампаний из Canvas
     * с кастомным timeout и нужным форматом
     */
    public List<UcCreativeResponse> getUcCreativeData(Long clientId, List<Long> creativeIds) {
        return getUcCreativeData(clientId, creativeIds, config.getReadTimeout());
    }

    /**
     * XXXXXXX
     */
    public PresetResponse getCreativeTemplates() {
        PresetResponse creativeResponses = doRequest(new GetPresetsRequest(), config.getCanvasUrl(),
                config.getReadTimeout(), GET);
        return creativeResponses;
    }

    /**
     * Получить закешированные outdoor разрешения на которые будут нарезаться креативы.
     * Обертка над getFfmpegResolutions(OUTDOOR_TYPE) с кешем.
     * Если канвас недоступен, то вернется пустой список.
     */
    public List<FfmpegResolutionsResponse> getCachedOutdoorFfmpegResolutions() {
        return outdoorFfmpegResolutionsCache.get();
    }

    /**
     * Получить разрешения на которые будут нарезаться креативы.
     */
    public List<FfmpegResolutionsResponse> getFfmpegResolutions(FfmpegResolutionsRequestType type) {
        GetFfmpegResolutionsRequest request = new GetFfmpegResolutionsRequest(type);
        return doRequest(request, config.getCanvasBackendUrl(), Duration.ofSeconds(5L), GET);
    }

    /**
     * Сгененрировать дефолтные video_addition-креативы из Canvas с кастомным по условиям
     */
    public List<List<Creative>> generateAdditions(Long clientId, List<GenerateConditions> conditions) {
        return doRequest(new GenerateVideoAdditions(clientId, conditions),
                config.getCanvasVideoUrl(), config.getReadTimeout(), POST);
    }

    /**
     * Загрузить видео по переданному урлу, подобрать подходящий шаблон креатива.
     * В случае uacVideoCreativeType будет использован VideoCreativeType.TEXT
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/canvas/src/main/java/ru/yandex/canvas/controllers/video
     * /VideoDefaultController.java?rev=r8508956#L261
     */
    public VideoUploadResponse createVideoFromUrl(
            Long clientId, String url, @Nullable UacVideoCreativeType uacVideoCreativeType, @Nullable Locale locale
    ) {
        return doRequest(new CreateVideoFromUrlRequest(clientId, url, uacVideoCreativeType, locale),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }

    /**
     * Загрузить видео с названием, подобрать подходящий шаблон креатива.
     * В случае uacVideoCreativeType будет использован VideoCreativeType.TEXT
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/canvas/src/main/java/ru/yandex/canvas/controllers/video
     * /VideoDefaultController.java?rev=r8508956#L261
     */
    public VideoUploadResponse createVideoFromFile(
            Long clientId, byte[] data, String name, @Nullable UacVideoCreativeType uacVideoCreativeType, @Nullable Locale locale
    ) {
        return doRequest(new CreateVideoFromFileRequest(clientId, data, name, uacVideoCreativeType, locale),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }

    /**
     * Получить данные по видео.
     */
    public VideoUploadResponse getCreatedVideo(
            Long clientId, Long presetId, String videoId,
            @Nullable UacVideoCreativeType uacVideoCreativeType) {
        return doRequest(new GetCreatedVideoRequest(clientId, presetId, videoId, uacVideoCreativeType),
                config.getCanvasBackendUrl(), config.getReadTimeout(), POST);
    }

    /**
     * Получить данные по видео.
     */
    public VideoResponse getVideo(Long clientId, Long presetId, String videoId) {
        return doRequest(new GetVideoRequest(clientId, presetId, videoId),
                config.getCanvasBackendUrl(), config.getReadTimeout(), POST);
    }

    /**
     * Получить данные о видеофайле
     */
    public VideoUploadResponse getVideoById(Long clientId, String videoId) {
        return doRequest(new GetVideoByIdRequest(clientId, videoId),
                config.getCanvasBackendUrl(), config.getReadTimeout(), GET);
    }

    /**
     * Создать дефолтный креатив по переданным шаблону и id загруженного видео.
     */
    public AdditionResponse createDefaultAddition(Long clientId, Long presetId, String videoId) {
        return doRequest(new CreateDefaultAdditionRequest(clientId, presetId, videoId),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }

    /**
     *  Загрузить html5-архив с названием
     */
    public Html5SourceResponse uploadHtml5(Long clientId, byte[] data, String name, Html5Tag html5Tag, @Nullable Locale locale) {
        return doRequest(new UploadHtml5Request(clientId, data, name, html5Tag, locale),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }

    /**
     * Создать html5 батч
     */
    public Html5BatchResponse createHtml5Batch(Long clientId, String name, String sourceId) {
        return doRequest(new CreateHtml5BatchRequest(clientId, name, sourceId),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }

    /**
     * Загрузить созданный креатив в Директ
     */
    public List<Long> uploadCreativeToDirect(Long clientId, Long userId, String batchId, Long creativeId) {
        return doRequest(new UploadHtml5CreativeToDirectRequest(clientId, userId, batchId, creativeId),
                config.getCanvasBackendUrl(), Duration.ofSeconds(45L), POST);
    }


    /**
     * Сделать запрос к ручке с заданным таймаутом и вернуть ее ответ в виде объекта ответа ручки,
     * заданного в ее спеке.
     * <p>
     * Бросает {@link CanvasClientException} при неудачной попытке распарсить ответ.
     */
    @Nonnull
    private <T> T doRequest(CanvasRequest<T> request, URI url, Duration timeout, String method) {
        Result<String> result;
        var metricsPath = request.getMetricsPath();
        if(metricsPath.endsWith("/")) {
            metricsPath = metricsPath.substring(0, metricsPath.length() - 1);
        }
        MetricRegistry parallelFetcherMetricRegistry =
                SolomonUtils.getParallelFetcherMetricRegistry(CANVAS_CLIENT + ":" + metricsPath);
        try (ParallelFetcher<String> fetcher = parallelFetcherFactory
                .getParallelFetcherWithMetricRegistry(parallelFetcherMetricRegistry);
             TraceProfile profile = Trace.current().profile(CANVAS_CLIENT, request.getPath());
             TraceChild traceChild = Trace.current().child(CANVAS_CLIENT, request.getPath())) {
            RequestBuilder builder = new RequestBuilder(method)
                    .setRequestTimeout(Ints.checkedCast(timeout.toMillis()))
                    .setUrl(buildUrl(url, request.getPath()))
                    .setCharset(StandardCharsets.UTF_8)
                    .addHeader(CONTENT_TYPE, request.getContentType())
                    .addHeader(X_YANDEX_TRACE, traceToHeader(traceChild))
                    .addHeader(AUTHORIZATION, config.getAuthToken());

            if (request.getLocale() != null) {
                builder.addHeader(ACCEPT_LANGUAGE, request.getLocale());
            }

            request.prepareRequest(builder);

            ParsableRequest<String> asyncRequest = new ParsableStringRequest(builder.build());

            logger.trace("Request body: {}", asyncRequest);
            result = fetcher.execute(asyncRequest);

            if (result.getErrors() != null && !result.getErrors().isEmpty()) {
                logger.error("Got errors: {}", result.getErrors());

                RuntimeException ex = new CanvasClientException("Got error on response for Canvas request "
                        + request
                        + "errors:" + result.getErrors(), getCanvasErrors(result)
                );

                result.getErrors().forEach(ex::addSuppressed);

                throw ex;
            }
        } catch (InterruptedException ex) {
            logger.error("Request were unexpectedly interrupted");
            Thread.currentThread().interrupt();
            throw new CanvasClientException(ex);
        }

        T response = request.deserializeResponse(result.getSuccess());
        if (response == null) {
            logger.error("Got unexpected response: \"{}\"", result.getSuccess());
            throw new CanvasClientException("Deserialization result is null");
        }
        return response;
    }

    private List<String> getCanvasErrors(Result<String> result) {
        List<String> canvasResponses = new ArrayList();

        if (result.getErrors() == null) {
            return canvasResponses;
        }

        result.getErrors().forEach((Throwable exception) -> {
            var errorWithResponse  =  safeCast(exception, ErrorResponseWrapperException.class);

            if (errorWithResponse != null && errorWithResponse.getResponse() != null && errorWithResponse.getResponse().hasResponseBody()) {
                var errorResponseBody = errorWithResponse.getResponse().getResponseBody();
                var videoValidationError = JsonUtils
                        .fromJsonIgnoringErrors(errorResponseBody, VideoValidationErrorResponse.class);

                if (videoValidationError != null && videoValidationError.getMessage() != null) {
                    canvasResponses.add(videoValidationError.getMessage());
                }

                var html5ValidationError = JsonUtils
                        .fromJsonIgnoringErrors(errorResponseBody, Html5ValidationErrorResponse.class);
                if (html5ValidationError != null && html5ValidationError.getMessages() != null) {
                    canvasResponses.addAll(html5ValidationError.getMessages());
                }
            }
        });

        return canvasResponses;
    }

    private String buildUrl(URI url, String path) {
        String realPath = url.getPath() +
                (path.startsWith(PATH_DELIMITER) ? path : PATH_DELIMITER + path);
        try {
            return new URIBuilder(url).setPath(realPath).build().toASCIIString();
        } catch (URISyntaxException e) {
            throw new CanvasClientException(e);
        }
    }

    /**
     * Кешируем результат getFfmpegResolutions(OUTDOOR_TYPE).
     * Если канвас недоступен, то возвращаем пустой список.
     */
    private Supplier<List<FfmpegResolutionsResponse>> createOutdoorFfmpegResolutionsCache() {
        return Suppliers.memoizeWithExpiration(() ->
        {
            try {
                return getFfmpegResolutions(OUTDOOR_TYPE);
            } catch (RuntimeException exception) {
                return Collections.emptyList();
            }
        }, 30, TimeUnit.SECONDS);
    }

}
