package ru.yandex.direct.canvas.client;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;

import org.apache.commons.io.FileUtils;

import ru.yandex.direct.canvas.client.model.exception.CanvasClientException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;

/**
 * Конфигурация клиента для Canvas
 */
public class CanvasClientConfiguration {
    private static final Duration DEFAULT_READ_TIMEOUT = Duration.ofSeconds(15);
    private static final String ENCODING = "UTF-8";
    private final String authToken;
    private final Duration readTimeout;
    private final URI canvasVideoUrl;
    private final URI canvasBackendUrl;
    private final URI canvasUrl;
    private final int chunkSize;

    public CanvasClientConfiguration(String videoUrl, String canvasBackendUrl, String canvasUrl,
                                     String tokenFile, String authToken, int chunkSize)
            throws URISyntaxException {
        this(videoUrl, canvasBackendUrl, canvasUrl, tokenFile, authToken, DEFAULT_READ_TIMEOUT, chunkSize);
    }

    /**
     * Бросает {@link CanvasClientException} при неудачной попытке получить authToken
     */
    public CanvasClientConfiguration(String videoUrl, String canvasBackendUrl, String canvasUrl,
                                     String tokenFilePath, String authToken,
                                     Duration readTimeout, int chunkSize) throws URISyntaxException {

        checkArgument(!isNullOrEmpty(videoUrl));
        checkArgument(!isNullOrEmpty(canvasUrl));
        checkNotNull(readTimeout);
        checkArgument(!(readTimeout.isNegative() || readTimeout.isZero()));
        canvasVideoUrl = new URI(videoUrl);
        this.canvasUrl = new URI(canvasUrl);

        if (authToken == null || authToken.isEmpty()) {
            authToken = getAuthTokenFromFile(tokenFilePath);
        }
        this.authToken = authToken;
        this.readTimeout = readTimeout;
        this.chunkSize = chunkSize;
        this.canvasBackendUrl = new URI(canvasBackendUrl);
    }


    public String getAuthToken() {
        return authToken;
    }

    public Duration getReadTimeout() {
        return readTimeout;
    }

    public URI getCanvasVideoUrl() {
        return canvasVideoUrl;
    }

    public URI getCanvasUrl() {
        return canvasUrl;
    }

    String getAuthTokenFromFile(String tokenFilePath) {
        try {
            return FileUtils.readFileToString(new File(tokenFilePath), ENCODING).trim();
        } catch (IOException e) {
            throw new CanvasClientException("failed to read auth_token from token_file: " + tokenFilePath);
        }
    }

    public int getChunkSize() {
        return chunkSize;
    }

    public URI getCanvasBackendUrl() {
        return canvasBackendUrl;
    }
}
