#include "ru_yandex_direct_clemmer_Clemmer.h"
#include "util.h"

#include <ads/clemmer/lib/clemmer.h>

class TClemmerResult {
    clemmer2_result result_;

public:
    TClemmerResult(clemmer2_result result)
        : result_(result)
    { }

    ~TClemmerResult() {
        Reset();
    }

    explicit operator bool() const {
        return result_;
    }

    clemmer2_result Get() const {
        return result_;
    }

    void Reset() {
        if (result_ != nullptr) {
            clemmer2_free_result(result_);
            result_ = nullptr;
        }
    }
};

class TClemmerConverter {
    JNIEnv* env;
    TJniClass word_class;
    TJniClass arraylist_class;
    jmethodID word_constructor;
    jmethodID arraylist_constructor;
    jmethodID arraylist_add;

public:
    TClemmerConverter(JNIEnv* env)
        : env(env)
        , word_class(env, "ru/yandex/direct/clemmer/ClemmerWord")
        , arraylist_class(env, "java/util/ArrayList")
        , word_constructor(word_class.GetMethodID("<init>", "(Ljava/lang/String;Ljava/util/List;Ljava/util/List;II)V"))
        , arraylist_constructor(arraylist_class.GetMethodID("<init>", "()V"))
        , arraylist_add(arraylist_class.GetMethodID("add", "(Ljava/lang/Object;)Z"))
    { }

    template<class T, class Func>
    TJniReference<jobject> MakeList(const T* item, Func&& func) {
        TJniReference<jobject> listobj(env, env->NewObject(arraylist_class, arraylist_constructor));
        if (!listobj || env->ExceptionCheck()) {
            throw TJavaException();
        }
        while (item != nullptr) {
            TJniReference<jobject> obj = func(item);
            env->CallBooleanMethod(listobj.Get(), arraylist_add, obj.Get());
            if (env->ExceptionCheck()) {
                throw TJavaException();
            }
            item = item->next;
        }
        return listobj;
    }

    TJniReference<jobject> MakeLemma(const clemmer2_lemma* lemma) {
        return jniMakeString(env, lemma->text);
    }

    TJniReference<jobject> MakeForma(const clemmer2_forma* forma) {
        return jniMakeString(env, forma->text);
    }

    TJniReference<jobject> MakeWord(const clemmer2_word* word) {
        TJniReference<jstring> javatext = jniMakeString(env, word->text);
        TJniReference<jobject> javalemmas = MakeList(word->lemmas, [&](const clemmer2_lemma* lemma){
            return MakeLemma(lemma);
        });
        TJniReference<jobject> javaformas = MakeList(word->formas, [&](const clemmer2_forma* forma){
            return MakeForma(forma);
        });
        jint flags = word->flags;
        jint raw_flags = word->raw_flags;
        TJniReference<jobject> obj(env, env->NewObject(
            word_class,
            word_constructor,
            javatext.Get(),
            javalemmas.Get(),
            javaformas.Get(),
            flags,
            raw_flags
        ));
        if (!obj || env->ExceptionCheck()) {
            throw TJavaException();
        }
        return obj;
    }

    TJniReference<jobject> MakeWordList(const clemmer2_word* wordlist) {
        return MakeList(wordlist, [&](const clemmer2_word* word){
            return MakeWord(word);
        });
    }
};

/*
 * Class:     ru_yandex_direct_clemmer_Clemmer
 * Method:    analyze2
 * Signature: (Ljava/lang/String;IZ)Ljava/util/List;
 */
JNIEXPORT jobject JNICALL Java_ru_yandex_direct_clemmer_Clemmer_analyze2(
    JNIEnv* env,
    jclass,
    jstring phrase,
    jint lang,
    jboolean fillformas)
{
    return jniWrapExceptions(env, [&]() -> jobject {
        TJniStringData phrase_data(env, phrase);
        if (!phrase_data) {
            throw TJavaException(env, "java/lang/IllegalArgumentException", "cannot get phrase data");
        }
        size_t phrase_size = ::strlen(phrase_data.Data());
        TClemmerResult cresult(
            fillformas
            ? clemmer2_analyze(phrase_data.Data(), phrase_size, static_cast<clemmer2_lang>(lang))
            : clemmer2_analyze_noformas(phrase_data.Data(), phrase_size, static_cast<clemmer2_lang>(lang))
        );
        if (!cresult) {
            throw TJavaException(env, "java/lang/IllegalArgumentException", "clemmer2_analyze failed with the given arguments");
        }
        phrase_data.Reset(); // no longer needed
        TClemmerConverter converter(env);
        TJniReference<jobject> wordlist = converter.MakeWordList(cresult.Get());
        return wordlist.Release();
    });
}
