#pragma once

#include <jni.h>
#include <memory>
#include <stdexcept>
#include <util/generic/string.h>
#include <util/generic/noncopyable.h>

/**
 * Throws a new Java exception is there's nothing currently being thrown
 */
inline void jniThrowException(JNIEnv* env, const char* classname, const char* message) {
    if (!env->ExceptionCheck()) {
        jclass clazz = env->FindClass(classname);
        if (!env->ExceptionCheck()) {
            env->ThrowNew(clazz, message);
            env->DeleteLocalRef(clazz);
        }
    }
}

/**
 * Marker class for passthru of existing Java exception
 */
class TJavaException {
public:
    TJavaException() {}
    TJavaException(JNIEnv* env, const char* classname, const char* message) {
        jniThrowException(env, classname, message);
    }
};

/**
 * Protects from C++ exceptions crossing the JNI boundary
 */
template<class Func>
inline auto jniWrapExceptions(JNIEnv* env, Func&& func) {
    try {
        return func();
    } catch (const TJavaException& e) {
        jniThrowException(env, "java/lang/IllegalStateException", "Unknown Java exception");
    } catch (const std::invalid_argument& e) {
        jniThrowException(env, "java/lang/IllegalArgumentException", e.what());
    } catch (const std::exception& e) {
        jniThrowException(env, "java/lang/RuntimeException", e.what());
    } catch (...) {
        jniThrowException(env, "java/lang/RuntimeException", "Unknown C++ exception");
    }
    return decltype(func())();
}

template<class T>
class TJniReference : public TNonCopyable {
    JNIEnv* env_;
    T value_;

public:
    TJniReference(JNIEnv* env, T value)
        : env_(env)
        , value_(value)
    { }

    template<class U>
    TJniReference(TJniReference<U>&& other) {
        env_ = other.Env();
        value_ = other.Release();
    }

    ~TJniReference() {
        Reset();
    }

    JNIEnv* Env() const {
        return env_;
    }

    T Get() const {
        return value_;
    }

    void Set(T value) {
        if (value != value_) {
            Reset();
            value_ = value;
        }
    }

    void Reset() {
        if (value_) {
            env_->DeleteLocalRef(value_);
            value_ = nullptr;
        }
    }

    T Release() {
        T value = value_;
        value_ = nullptr;
        return value;
    }

    operator T() const {
        return value_;
    }

    explicit operator bool() const {
        return value_;
    }
};

class TJniClass : public TJniReference<jclass> {
public:
    TJniClass(JNIEnv* env, const char* name)
        : TJniReference(env, env->FindClass(name))
    {
        if (!*this) {
            throw TJavaException();
        }
    }

    jmethodID GetMethodID(const char* name, const char* sig) {
        jmethodID id = Env()->GetMethodID(Get(), name, sig);
        if (!id) {
            throw TJavaException();
        }
        return id;
    }
};

class TJniStringData {
    JNIEnv* env_;
    jstring str_;
    const char* data_;

public:
    TJniStringData(JNIEnv* env, jstring str)
        : env_(env)
        , str_(str)
    {
        data_ = env->GetStringUTFChars(str, nullptr);
        if (!data_) {
            throw TJavaException();
        }
    }

    ~TJniStringData() {
        Reset();
    }

    explicit operator bool() const {
        return data_;
    }

    const char* Data() const {
        return data_;
    }

    void Reset() {
        if (data_) {
            env_->ReleaseStringUTFChars(str_, data_);
            data_ = nullptr;
        }
    }
};

inline TJniReference<jstring> jniMakeString(JNIEnv* env, const char* text) {
    jstring str = env->NewStringUTF(text);
    if (!str) {
        throw TJavaException();
    }
    return TJniReference<jstring>(env, str);
}

inline TJniReference<jstring> jniMakeString(JNIEnv* env, const char* text, size_t size) {
    TString buffer(text, size);
    jstring str = env->NewStringUTF(buffer.c_str());
    if (!str) {
        throw TJavaException();
    }
    return TJniReference<jstring>(env, str);
}
