package ru.yandex.direct.clickhouse;

import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.ZoneId;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Charsets;
import org.apache.commons.io.IOUtils;

import ru.yandex.direct.process.Docker;
import ru.yandex.direct.process.DockerContainer;
import ru.yandex.direct.process.DockerHostUserEntryPoint;
import ru.yandex.direct.process.DockerNetwork;
import ru.yandex.direct.process.DockerRunner;
import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.Transient;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Класс для работы с ClickHouse-кластером из Docker-контейнеров.
 * <p>
 * Если нужен способ запустить кластер из junit-тестов - см. {@code ClickHouseClusterForTest}
 * в direct/libs/test-clickhouse.
 */
@ParametersAreNonnullByDefault
public class ClickHouseCluster implements AutoCloseable {
    private static final String ZOOKEEPER_IMAGE = "zookeeper:3.4";

    private final String clusterName;
    private final AutoCloseableList<ClickHouseContainer> clickHouses;
    private final AutoCloseableList<DockerContainer> zookeepers;
    private final DockerNetwork network;
    private final boolean useIpv6;

    @SuppressWarnings("checkstyle:parameternumber")
    ClickHouseCluster(TempDirectory generatedConfigsDir, Docker docker, String clusterName, @Nullable Path dataRootPath,
                      Map<Integer, String> zookeeperHosts, Map<String, Path> clickHousesWithConfigs, boolean useIpv6,
                      Map<String, Integer> httpPorts, Map<String, Integer> nativePorts)
            throws InterruptedException {
        this.clusterName = clusterName;
        this.useIpv6 = useIpv6;
        if (dataRootPath != null) {
            Checked.run(() -> Files.createDirectories(dataRootPath));
        }
        try (
                Transient<DockerNetwork> transientNetwork = new Transient<>(new DockerNetwork(docker, clusterName));
                Transient<AutoCloseableList<DockerContainer>> transientZookeepers = new Transient<>(
                        new AutoCloseableList<>());
                Transient<AutoCloseableList<ClickHouseContainer>> transientClickHouses = new Transient<>(
                        new AutoCloseableList<>())
        ) {
            createZookeepers(transientZookeepers.item, clusterName, docker, transientNetwork.item,
                    dataRootPath, generatedConfigsDir, zookeeperHosts);
            createClickHouses(transientClickHouses.item, clusterName, docker, transientNetwork.item,
                    dataRootPath, generatedConfigsDir, clickHousesWithConfigs, this.useIpv6,
                    httpPorts, nativePorts);
            this.network = transientNetwork.pop();
            this.zookeepers = transientZookeepers.pop();
            this.clickHouses = transientClickHouses.pop();
        }
    }

    private static void createZookeepers(AutoCloseableList<DockerContainer> zookeepers,
                                         String clusterName, Docker docker, DockerNetwork network,
                                         @Nullable Path dataRootPath,
                                         TempDirectory generatedConfigsDir, Map<Integer, String> zookeeperHosts)
            throws InterruptedException {
        String confSubpath = "conf";
        String dataSubpath = "data";
        String dataLogSubpath = "data_log";
        for (Map.Entry<Integer, String> entry : zookeeperHosts.entrySet()) {
            int zooId = entry.getKey();
            String hostname = entry.getValue();
            DockerRunner dockerRunner = new DockerRunner(docker, ZOOKEEPER_IMAGE)
                    .withName(clusterName + "_" + hostname)
                    .withHostname(hostname)
                    .withNetworkAlias(hostname)
                    .withRestartAlways()
                    .withNetwork(network)
                    .withEnvironment("ZOO_MY_ID", Integer.toString(zooId))
                    .withEnvironment("ZOO_SERVERS", zookeeperHosts.entrySet().stream()
                            .sorted(Comparator.comparing(Map.Entry::getKey))
                            .map(e -> String.format("server.%d=%s_%s:2888:3888",
                                    e.getKey(), clusterName, e.getValue()))
                            .collect(Collectors.joining(" "))
                    );
            if (dataRootPath != null) {
                Path hostVolumePath = dataRootPath.resolve(hostname).toAbsolutePath();

                // zookeeper не запустится, если предварительно не создать для него каталоги conf, data и data_log.
                for (String subpath : Arrays.asList(confSubpath, dataSubpath, dataLogSubpath)) {
                    Checked.run(() -> Files.createDirectories(hostVolumePath.resolve(subpath)));
                }
                Path containerVolumePath = Paths.get("/volume");  // Можно менять, передаётся через переменные окружения

                // Несмотря на то, что исходный entrypoint от zookeeper смотрит в переменную ZOO_CONF_DIR,
                // следующий скрипт упорно пытается читать из /conf. Поэтому вместо установки переменной
                // окружения цепляется отдельный volume на каталог для конфигов. При запуске /docker-entrypoint.sh
                // создаёт в этом каталоге zoo.cfg, с использованием которого запускается zookeeper.
                Path confContainerPath = Paths.get("/conf");
                dockerRunner = new DockerHostUserEntryPoint(generatedConfigsDir, "/docker-entrypoint.sh")
                        .apply(dockerRunner)
                        .withCmd("zkServer.sh", "start-foreground")
                        .withEnvironment("ZOO_DATA_DIR", containerVolumePath.resolve(dataSubpath).toString())
                        .withEnvironment("ZOO_DATA_LOG_DIR", containerVolumePath.resolve(dataLogSubpath).toString())
                        .withEnvironment("ZOO_USER", System.getProperty("user.name"))
                        .withVolume(hostVolumePath, containerVolumePath, false)
                        .withVolume(hostVolumePath.resolve(confSubpath), confContainerPath, false);
            }
            try (Transient<DockerContainer> zookeeper = new Transient<>(new DockerContainer(dockerRunner))) {
                zookeepers.add(zookeeper);
            }
        }
    }

    @SuppressWarnings({
            // Хардкод путей в контейнерах и IP-адреса
            "squid:S1075", "squid:S1313",
            "checkstyle:parameternumber"
    })
    private static void createClickHouses(AutoCloseableList<ClickHouseContainer> clickHouseContainers,
                                          String clusterName, Docker docker, DockerNetwork network,
                                          @Nullable Path dataRootPath, TempDirectory generatedConfigsDir,
                                          Map<String, Path> clickHousesWithConfigs, boolean hasIpv6,
                                          Map<String, Integer> httpPorts, Map<String, Integer> nativePorts)
            throws InterruptedException {
        for (Map.Entry<String, Path> entry : clickHousesWithConfigs.entrySet()) {
            String hostname = entry.getKey();
            Path metrikaXmlHostPath = entry.getValue();

            // В образе yandex/clickhouse-server запуск clickhouse-сервера занесён не в секцию cmd, а в секцию
            // entrypoint, что можно считать ошибкой - такое поведение не позволяет стартовать контейнер с произвольной
            // командой. Поэтому entrypoint очищается, а всё, что он делал, перенесено в cmd.
            DockerRunner dockerRunner = new DockerRunner(docker, ClickHouseContainer.CLICKHOUSE_IMAGE)
                    .withEntrypoint("")
                    .withEnvironment("TZ", ZoneId.systemDefault().getId())
                    .withName(clusterName + "_" + hostname)
                    .withHostname(hostname)
                    .withNetworkAlias(hostname)
                    .withNetwork(network)
                    .withVolume(metrikaXmlHostPath, Paths.get("/etc/metrika.xml"), true);

            try {
                Path cmdScriptHost = Files.createTempFile(
                        generatedConfigsDir.getPath(),
                        "clickhouse_cmd.",
                        ".sh",
                        PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwxr-xr-x")));
                try (InputStream scriptStream = ClickHouseCluster.class
                        .getResourceAsStream("clickhouse_cmd.sh");
                     Writer writer = new FileWriter(cmdScriptHost.toString())) {
                    IOUtils.copy(scriptStream, writer, Charsets.UTF_8);
                }
                dockerRunner = dockerRunner
                        .withVolume(cmdScriptHost, Paths.get("/cmd.sh"), true);
            } catch (IOException e) {
                throw new Checked.CheckedException(e);
            }

            if (!hasIpv6) {
                dockerRunner = dockerRunner.withCmd("/bin/bash", "/cmd.sh", "--no-ipv6");
            } else {
                dockerRunner = dockerRunner.withCmd("/bin/bash", "/cmd.sh");
            }

            if (dataRootPath != null) {
                // clickhouse не запустится без этих каталогов.
                for (String subdir : Arrays.asList(".", "data")) {
                    Path path = dataRootPath.resolve(hostname).resolve(subdir);
                    Checked.run(() -> Files.createDirectories(path));
                }

                // Если хочется поменять этот каталог, то надо прописать его в clickhouse-server.xml, который пока что
                // не генерируется этим кодом.
                Path containerVolumePath = Paths.get("/var/lib/clickhouse");

                dockerRunner = new DockerHostUserEntryPoint(generatedConfigsDir, null)
                        .apply(dockerRunner)
                        .withVolume(dataRootPath.resolve(hostname).toAbsolutePath(),
                                containerVolumePath, false);
            }
            ClickHouseContainer.Builder clickHouseBuilder = ClickHouseContainer.builder()
                    .withDockerRunner(dockerRunner)
                    .withSoftStop(dataRootPath != null);
            Integer httpPort = httpPorts.get(hostname);
            if (httpPort != null) {
                clickHouseBuilder.withHostHttpPort(httpPort);
            }
            Integer nativePort = nativePorts.get(hostname);
            if (nativePort != null) {
                clickHouseBuilder.withHostNativePort(nativePort);
            }

            // Если нет нужды в повторном запуске, то незачем тратить время на корректное выключение clickhouse.
            try (Transient<ClickHouseContainer> clickHouse = new Transient<>(clickHouseBuilder.build())) {
                clickHouseContainers.add(clickHouse);
            }
        }
    }

    public boolean isUseIpv6() {
        return useIpv6;
    }

    /**
     * @param dbName Имя базы данных. База данных <b>не создаётся</b>, следует предварительно создать с помощью
     *               {@link #createDatabaseIfNotExists(String)}
     * @return Ключ - название хоста, значение - jdbc url, который можно передать в
     * {@link ru.yandex.clickhouse.ClickHouseDataSource#ClickHouseDataSource(String)}.
     */
    public Map<String, String> getClickHouseJdbcUrls(String dbName) {
        return getClickHousesStream()
                .collect(Collectors.toMap(
                        ClickHouseContainer::getHostname,
                        c -> String.format("jdbc:clickhouse://%s:%d/%s",
                                useIpv6 ? "[::1]" : "127.0.0.1",
                                c.getHttpHostPort().getPort(),
                                dbName)));
    }

    public Map<String, String> getClickHouseJdbcUrls() {
        return getClickHouseJdbcUrls("default");
    }

    public String getClusterName() {
        return clusterName;
    }

    public DockerNetwork getNetwork() {
        return network;
    }

    public Stream<ClickHouseContainer> getClickHousesStream() {
        return clickHouses.stream();
    }

    @Override
    public void close() {
        // Ресурсы нужно очищать в порядке last in - first out. Сначала плавно останавливаются clickhouse-контейнеры,
        // потом резко останавливаются zookeeper-контейнеры (zookeeper живучий, его можно резко выключать) и лишь потом
        // удаляется каталог с конфигурационными файлами.
        try (
                DockerNetwork ignored1 = network;
                AutoCloseableList<DockerContainer> ignored2 = zookeepers
        ) {
            clickHouses.close();
        }
    }

    /**
     * Ждать до тех пор, пока все clickhouse не будут готовы обрабатывать запросы.
     *
     * @param timeout Максимальное время ожидания.
     * @throws InterruptedException
     */
    public void awaitConnectivity(Duration timeout) throws InterruptedException {
        Completer.Builder completerBuilder = new Completer.Builder(Duration.ofSeconds(10));
        clickHouses.forEach(c -> completerBuilder.submitVoid(
                c.getHostname(),
                () -> c.awaitConnectivity(timeout)
        ));
        try (Completer completer = completerBuilder.build()) {
            completer.waitAll();
        }
    }

    public void createDatabaseIfNotExists(String dbName) throws SQLException {
        for (ClickHouseContainer container : clickHouses) {
            try (Connection connection = ClickHouseUtils.connect(container.getHttpHostPort());
                 Statement statement = connection.createStatement()) {
                statement.executeUpdate("CREATE DATABASE IF NOT EXISTS " + ClickHouseUtils.quoteName(dbName));
            }
        }
    }

    /**
     * Прочитать stderr всех контейнеров.
     *
     * @param tail    Параметры обрезки логов.
     * @param timeout Максимальное время ожидания ответа от Docker
     * @return Ключ - имя хоста, значение - лог одной строкой, если он был получен.
     */
    public Map<String, Optional<String>> readContainersStderr(DockerContainer.Tail tail, Duration timeout)
            throws InterruptedException {
        Completer.Builder completerBuilder = new Completer.Builder(timeout.plusSeconds(10));
        List<Future<AbstractMap.SimpleEntry<String, Optional<String>>>> futures =
                Stream.concat(zookeepers.stream(), clickHouses.stream())
                        .map(c -> completerBuilder.submit(
                                "read_stderr_" + c.getHostname(),
                                () -> new AbstractMap.SimpleEntry<>(
                                        c.getHostname(),
                                        c.tryReadStderr(tail, timeout))))
                        .collect(Collectors.toList());
        try (Completer completer = completerBuilder.build()) {
            completer.waitAll();
        }
        return futures.stream()
                .map(Checked.function(Future::get))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }
}
