package ru.yandex.direct.clickhouse;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.tuple.Pair;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import ru.yandex.direct.process.Docker;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Конфигуратор для запуска тестового кластера ClickHouse.
 * <p>
 * Пример.
 * <p>
 * Представим следующую конфигурацию кластера.
 * <pre>
 * {@code ClickHouseClusterBuilder builder = new ClickHouseClusterBuilder()
 *       .withZooKeeper(1, "zookeeper01")
 *       .withZooKeeper(2, "zookeeper02")
 *       .withZooKeeper(3, "zookeeper03")
 *       .withClickHouse("clickhouse01")
 *       .withClickHouse("clickhouse02")
 *       .withClickHouse("clickhouse03")
 *       .withClickHouse("clickhouse04")
 *       .withMacroLabel("macrolabel");
 *   builder.shardGroup()
 *       .withShard(1, "clickhouse01", "clickhouse02")
 *       .withShard(2, "clickhouse03", "clickhouse04");
 * }
 * </pre>
 * <p>
 * Предположим, на каждом сервере ClickHouse выполнили следующий SQL:
 * <pre>
 * {@code CREATE TABLE log ENGINE = TinyLog() AS SELECT <номер> AS id;
 *   CREATE TABLE log_distributed (id Int64) ENGINE = Distributed(macrolabel, default, log);
 *   CREATE TABLE merge ENGINE = ReplicatedMergeTree('/clickhouse/tables/{macrolabel}/merge', '{replica}',
 *       date, id, 1024) AS SELECT today() AS date, <номер> AS id;
 * }
 * </pre>
 * <p>
 * При соединении с <code>clickhouse01</code> результаты запросов будут следующими:
 * <ul>
 * <li><code>{@code SELECT id FROM log}</code> ожидаемо вернёт 1.</li>
 * <li><code>{@code SELECT id FROM log_distributed}</code> вернёт 1 и 3 - записи из первых реплик в каждом шарде.</li>
 * <li><code>{@code SELECT id FROM merge}</code> вернёт 1 и 2. При вставке значения в одну реплику запись копируется
 * на все остальные реплики. Сначала в clickhouse01 была вставлена запись 1, потом в clickhouse02 была вставлена
 * запись 2, потом clickhouse02 записал 2 в clickhouse01.</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
public class ClickHouseClusterBuilder {
    /**
     * zk index -> zk hostname
     */
    private final Map<Integer, String> zookeeperHosts;
    private final ShardGroup shardGroup;
    private final List<String> clickHouseHosts;
    private final Set<String> allHostNames;
    private final Map<String, Integer> httpPorts;
    private final Map<String, Integer> nativePorts;
    @Nullable
    private Boolean forceIpv6;
    private String macroLabel;

    @JsonCreator
    private ClickHouseClusterBuilder(@JsonProperty("zookeeper_hosts") Map<Integer, String> zookeeperHosts,
                                     @JsonProperty("clickhouse_hosts") List<String> clickhouseHosts,
                                     @JsonProperty("shards") ShardGroup shardGroup,
                                     @JsonProperty("http_ports") Map<String, Integer> httpPorts,
                                     @JsonProperty("native_ports") Map<String, Integer> nativePorts) {
        this.httpPorts = httpPorts;
        this.nativePorts = nativePorts;
        this.allHostNames = new HashSet<>();
        this.macroLabel = "shard";
        this.zookeeperHosts = zookeeperHosts;
        this.shardGroup = shardGroup;
        this.clickHouseHosts = clickhouseHosts;
        for (String host : zookeeperHosts.values()) {
            checkAndAddHost(host);
        }
        for (String host : clickHouseHosts) {
            checkAndAddHost(host);
        }
    }

    public ClickHouseClusterBuilder() {
        this(new HashMap<>(), new ArrayList<>(), new ShardGroup(), new HashMap<>(), new HashMap<>());
    }

    /**
     * Обработчик для {@link Map#merge(Object, Object, BiFunction)}, который не позволяет перезаписывать.
     */
    private static <A, B, C> BiFunction<A, B, C> noRewrite(String format, Object... args) {
        return (l, r) -> {
            throw new IllegalArgumentException(String.format(format, args));
        };
    }

    @JsonGetter("zookeeper_hosts")
    @SuppressWarnings("unused")
    public Map<Integer, String> getZookeeperHosts() {
        return zookeeperHosts;
    }

    @JsonGetter("clickhouse_hosts")
    @SuppressWarnings("unused")
    public List<String> getClickHouseHosts() {
        return clickHouseHosts;
    }

    @JsonGetter("http_ports")
    @SuppressWarnings("unused")
    public Map<String, Integer> getHttpPorts() {
        return httpPorts;
    }

    @JsonGetter("native_ports")
    @SuppressWarnings("unused")
    public Map<String, Integer> getNativePorts() {
        return nativePorts;
    }

    /**
     * Зарегистрировать сервер ZooKeeper.
     *
     * @param index    Индекс сервера. См.
     *                 https://clickhouse.yandex/docs/en/single/index.html#replicatedsummingmergetree
     * @param hostname Имя хоста, которое будет передано в контейнер. Также суффикс для названия контейнера.
     */
    public ClickHouseClusterBuilder withZooKeeper(int index, String hostname) {
        checkAndAddHost(hostname);
        zookeeperHosts.merge(index, hostname, noRewrite("ZooKeeper with index %d already exists", index));
        return this;
    }

    /**
     * Зарегистрировать сервер ClickHouse.
     *
     * @param hostname Имя хоста, которое будет передано в контейнер. Также суффикс для названия контейнера.
     */
    public ClickHouseClusterBuilder withClickHouse(String hostname) {
        checkAndAddHost(hostname);
        clickHouseHosts.add(hostname);
        return this;
    }

    public ClickHouseClusterBuilder withHttpPort(String hostname, int hostPort) {
        httpPorts.put(hostname, hostPort);
        return this;
    }

    public ClickHouseClusterBuilder withNativePort(String hostname, int hostPort) {
        nativePorts.put(hostname, hostPort);
        return this;
    }

    /**
     * Запустить кластер.
     *
     * @param tmpDir       Временный каталог, в который будут складываться сгенерированные конфигурационные файлы.
     * @param docker       Обёртка для работы с Docker
     * @param clusterName  Название кластера. Является префиксом для названия docker-контейнеров.
     * @param dataRootPath Путь на хосте, в котором будут находиться все базы данных. Может быть null, в таком случае
     *                     будет создан одноразовый кластер, все данные после завершения работы исчезнут.
     * @return Объект {@link ClickHouseCluster}, при закрытии которого контейнеры будут остановлены и удалены.
     * @throws IOException          Невозможно записать конфигурационные файлы в {@literal tmpDir}
     * @throws InterruptedException
     */
    public ClickHouseCluster build(TempDirectory tmpDir, Docker docker, String clusterName,
                                   @Nullable Path dataRootPath) throws IOException, InterruptedException {
        if (!shardGroup.isEmpty() && zookeeperHosts.isEmpty()) {
            throw new IllegalStateException("No ZooKeeper servers specified"
                    + " but it is required for distributed and replicated tables");
        }
        if (clickHouseHosts.isEmpty()) {
            throw new IllegalStateException("No ClickHouse servers specified");
        }
        validateGroups();
        validatePorts();
        try {
            Map<String, Node> metrikaConfigs = makeMetrikaConfigs();
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");

            Map<String, Path> clickHousesWithConfigs = new HashMap<>();
            for (Map.Entry<String, Node> metrikaConfig : metrikaConfigs.entrySet()) {
                String hostname = metrikaConfig.getKey();
                Path configPath = Files.createTempFile(tmpDir.getPath(),
                        String.format("metrika_%s_%s.", clusterName, hostname), ".xml",
                        PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwxr-xr-x")));
                transformer.transform(new DOMSource(metrikaConfig.getValue()), new StreamResult(configPath.toString()));
                clickHousesWithConfigs.put(hostname, configPath);
            }
            boolean useIpv6;
            if (forceIpv6 == null) {
                useIpv6 = docker.isDockerSupportsIpv6(ClickHouseContainer.CLICKHOUSE_IMAGE);
            } else {
                useIpv6 = forceIpv6;
            }
            return new ClickHouseCluster(tmpDir, docker, clusterName, dataRootPath, zookeeperHosts,
                    clickHousesWithConfigs, useIpv6, httpPorts, nativePorts);
        } catch (TransformerException | ParserConfigurationException e) {
            throw new Checked.CheckedException(e);
        }
    }

    private void checkAndAddHost(String hostname) {
        if (allHostNames.contains(hostname)) {
            throw new IllegalArgumentException("Cluster already contains hostname " + hostname);
        }
        allHostNames.add(hostname);
    }

    @SuppressWarnings("squid:S3776")  // cognitive complexity
    private void validateGroups() {
        for (Map.Entry<Integer, List<String>> shardEntry : shardGroup.shards.entrySet()) {
            for (String host : shardEntry.getValue()) {
                if (!allHostNames.contains(host)) {
                    throw new IllegalStateException(String.format(
                            "Shard %d of ReplicatedMergeTree group contains unregistered host %s",
                            shardEntry.getKey(), host));
                }
            }
        }
    }

    private void validatePorts() {
        Set<Integer> registeredPorts = new HashSet<>();
        Arrays.asList(Pair.of(httpPorts, "http"), Pair.of(nativePorts, "native")).forEach(mapAndLabel -> {
            mapAndLabel.getLeft().forEach((host, port) -> {
                if (!allHostNames.contains(host)) {
                    throw new IllegalStateException(String.format(
                            "Requested %s port %d for undefined host %s",
                            mapAndLabel.getRight(), port, host));
                }
                if (!registeredPorts.add(port)) {
                    throw new IllegalStateException("Duplicate request for port " + port);
                }
            });
        });
    }

    private Map<String, Node> makeMetrikaConfigs() throws ParserConfigurationException {
        Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
        Map<String, Node> result = new HashMap<>();
        Node distributedTableConfig = null;
        Node zookeeperConfig = null;

        Map<String, Node> macrosConfigs = makeMacrosConfigs(doc);

        for (String clickHouseHost : clickHouseHosts) {
            Node root = doc.createElement("yandex");

            if (distributedTableConfig == null) {
                distributedTableConfig = root.appendChild(makeDistributedTableConfig(doc));
            } else {
                root.appendChild(distributedTableConfig.cloneNode(true));
            }

            if (!zookeeperHosts.isEmpty()) {
                if (zookeeperConfig == null) {
                    zookeeperConfig = root.appendChild(makeZookeeperConfig(doc));
                } else {
                    root.appendChild(zookeeperConfig.cloneNode(true));
                }
            }

            macrosConfigs.computeIfPresent(clickHouseHost, (k, v) -> root.appendChild(v));
            result.put(clickHouseHost, root);
        }
        return result;
    }

    private Map<String, Node> makeMacrosConfigs(Document doc) {
        Map<String, Node> result = new HashMap<>();
        for (Map.Entry<Integer, List<String>> shardEntry : shardGroup.shards.entrySet()) {
            for (String clickHouseHost : shardEntry.getValue()) {
                String shardId = String.format("%02d", shardEntry.getKey());
                Node macros = result.computeIfAbsent(clickHouseHost, host -> {
                    Node m = doc.createElement("macros");
                    m.appendChild(doc.createElement("replica")).setTextContent(host);
                    return m;
                });
                macros.appendChild(doc.createElement(macroLabel)).setTextContent(shardId);
            }
        }
        return result;
    }

    private Node makeZookeeperConfig(Document doc) {
        Element root = doc.createElement("zookeeper-servers");
        for (Map.Entry<Integer, String> entry : zookeeperHosts.entrySet()) {
            Element node = (Element) root.appendChild(doc.createElement("node"));
            node.setAttribute("index", Integer.toString(entry.getKey()));
            node.appendChild(doc.createElement("host")).setTextContent(entry.getValue());
            node.appendChild(doc.createElement("port")).setTextContent("2181");
        }
        return root;
    }

    private Node makeDistributedTableConfig(Document doc) {
        Element root = doc.createElement("clickhouse_remote_servers");
        Node groupNode = root.appendChild(doc.createElement(macroLabel));
        for (Collection<String> shard : shardGroup.getShards().values()) {
            Node shardNode = groupNode.appendChild(doc.createElement("shard"));
            shardNode.appendChild(doc.createElement("weight")).setTextContent("1");
            shardNode.appendChild(doc.createElement("internal_replication")).setTextContent("true");
            for (String clickHouseHost : shard) {
                Node replicaNode = shardNode.appendChild(doc.createElement("replica"));
                replicaNode.appendChild(doc.createElement("host")).setTextContent(clickHouseHost);
                replicaNode.appendChild(doc.createElement("port")).setTextContent("9000");
            }
        }
        return root;
    }

    /**
     * Создать конфигурацию для таблиц с движками Replicated*MergeTree.
     * См. https://clickhouse.yandex/docs/en/single/index.html#creating-replicated-tables
     * <p>
     * Также будет создана конфигурация таблиц Distributed путём транспонирования матрицы шард-реплика.
     * См. https://clickhouse.yandex/docs/en/single/index.html#distributed
     *
     * @return Конфигуратор
     */
    @JsonGetter("shards")
    public ShardGroup shardGroup() {
        return shardGroup;
    }

    /**
     * Использовать ли в контейнерах IPv6
     *
     * @param forceIpv6 <ul>
     *                  <li>{@code true} - принудительно включить IPv6</li>
     *                  <li>{@code false} - принудительно отключить IPv6</li>
     *                  <li>{@code null} - автоматически определить, поддерживается ли IPv6 и включить, если
     *                  поддерживается. Поведение по умолчанию.</li>
     *                  </ul>
     */
    public ClickHouseClusterBuilder forceIpv6(@Nullable Boolean forceIpv6) {
        this.forceIpv6 = forceIpv6;
        return this;
    }

    /**
     * Метка для секции {@literal <macro>} в конфиге ClickHouse. По умолчанию "shard".
     * См. https://clickhouse.yandex/docs/en/single/index.html#creating-replicated-tables
     */
    public ClickHouseClusterBuilder withMacroLabel(String macroLabel) {
        this.macroLabel = macroLabel;
        return this;
    }

    public static class ShardGroup {
        private final Set<String> registeredHosts = new HashSet<>();
        private final Map<Integer, List<String>> shards;

        @JsonCreator
        private ShardGroup(@JsonProperty("shards") Map<Integer, List<String>> shards) {
            this.shards = new HashMap<>();
            for (Map.Entry<Integer, List<String>> entry : shards.entrySet()) {
                withShard(entry.getKey(), entry.getValue());
            }
        }

        ShardGroup() {
            this(new HashMap<>());
        }

        List<String> register(List<String> nodes) {
            List<String> shard = new ArrayList<>();
            for (String node : nodes) {
                if (registeredHosts.contains(node)) {
                    throw new IllegalArgumentException(String.format("Node %s figured in two shards", node));
                }
                registeredHosts.add(node);
                shard.add(node);
            }
            return shard;
        }

        @JsonIgnore
        public boolean isEmpty() {
            return shards.isEmpty();
        }

        @JsonGetter("shards")
        @SuppressWarnings("unused")
        public Map<Integer, List<String>> getShards() {
            return shards;
        }

        /**
         * Зарегистрировать новый шард.
         * См. https://clickhouse.yandex/docs/en/single/index.html#creating-replicated-tables
         *
         * @param index           Индекс шарда.
         * @param clickHouseHosts Список всех ClickHouse-реплик в шарде. Реплики предварительно должны быть
         *                        зарегистрированы с помощью {@link #withClickHouse(String)}.
         */
        public ShardGroup withShard(int index, String... clickHouseHosts) {
            return withShard(index, Arrays.asList(clickHouseHosts));
        }

        private ShardGroup withShard(int index, List<String> clickHouseHosts) {
            shards.merge(index, register(clickHouseHosts),
                    noRewrite("ReplicatedMergeTree shard %d already exists", index));
            return this;
        }
    }
}
