package ru.yandex.direct.clickhouse;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.sql.Statement;
import java.time.Duration;
import java.util.Objects;

import com.google.common.primitives.Ints;

import ru.yandex.clickhouse.ClickHouseConnection;
import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.clickhouse.except.ClickHouseUnknownException;
import ru.yandex.clickhouse.settings.ClickHouseProperties;
import ru.yandex.direct.process.Docker;
import ru.yandex.direct.process.DockerContainer;
import ru.yandex.direct.process.DockerRunner;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.HostPort;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.Transient;
import ru.yandex.direct.utils.db.DbInstance;

public class ClickHouseContainer extends DockerContainer implements DbInstance {
    public static final String CLICKHOUSE_IMAGE = "yandex/clickhouse-server:19.17";
    public static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5);
    public static final Duration DEFAULT_STOP_TIMEOUT = Duration.ofSeconds(5);

    private final HostPort httpHostPort;
    private final HostPort nativeHostPort;
    private final boolean softStop;

    private ClickHouseContainer(DockerRunner runner, boolean softStop) throws InterruptedException {
        super(runner);
        this.softStop = softStop;
        try (Transient<DockerContainer> guard = new Transient<>(this)) {
            httpHostPort = getPublishedPort(8123);
            nativeHostPort = getPublishedPort(9000);
            guard.success();
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public HostPort getHttpHostPort() {
        return httpHostPort;
    }

    public HostPort getNativeHostPort() {
        return nativeHostPort;
    }

    @Override
    public ClickHouseConnection connect() throws SQLException {
        return connect(DEFAULT_CONNECT_TIMEOUT);
    }

    @Override
    public void close() {
        if (softStop) {
            // Если clickhouse вырубить жёстко, то он может не удалить свою запись в zookeeper. Хоть в zookeeper
            // хранится и эфемерная нода, zookeeper может обнаружить пропажу клиента через значительный промежуток
            // времени. При повторном запуске контейнера возникнет коллизия и кластер некоторое время будет отказывать
            // в обслуживании. Плавная остановка по sigterm решает эту проблему.
            MonotonicTime deadline = NanoTimeClock.now().plus(DEFAULT_STOP_TIMEOUT);
            try {
                stop(DEFAULT_STOP_TIMEOUT);
            } finally {
                // Гонка clickhouse в docker: после получения sigterm у контейнера в ExitStatus может быть код 0 или
                // 255, но на целостность кластера это не влияет. Сложно выяснить, завершился ли процесс успешно,
                // поэтому код возврата игнорируется.
                getDocker().checkContainerIdUninterruptibly(
                        deadline.minus(NanoTimeClock.now()),
                        getContainerId(),
                        "rm", "--force", "--volumes", getContainerId().toString());
            }
        } else {
            // clickhouse можно останавливать жёстко, если его дальнейшая судьба не важна.
            super.close();
        }
    }

    public ClickHouseDataSource getDataSource() {
        return getDataSource("default");
    }

    public ClickHouseDataSource getDataSource(String dbName) {
        if (dbName.isEmpty()) {
            throw new IllegalArgumentException("Empty dbName");
        }
        return new ClickHouseDataSource(String.format("jdbc:clickhouse://%s:%d/%s",
                getHttpHostPort().getHost(),
                getHttpHostPort().getPort(),
                dbName
        ));
    }

    @Override
    public ClickHouseConnection connect(Duration timeout) throws SQLException {
        ClickHouseProperties props = new ClickHouseProperties();
        props.setConnectionTimeout(Ints.saturatedCast(timeout.toMillis()));
        ClickHouseDataSource dataSource = new ClickHouseDataSource("jdbc:clickhouse://" + getHttpHostPort(), props);
        return (ClickHouseConnection) dataSource.getConnection();
    }

    public void awaitConnectivity(Duration timeout) throws InterruptedException {
        MonotonicTime deadline = NanoTimeClock.now().plus(timeout);
        Interrupts.InterruptibleConsumer<Throwable> handleError = e -> {
            Duration remainingTimeout = deadline.minus(NanoTimeClock.now());

            if (remainingTimeout.isNegative()) {
                throw new IllegalStateException("Failed to connect to clickhouse container", e);
            }
            Thread.sleep(CommonUtils.min(Duration.ofMillis(500), remainingTimeout).toMillis());

            checkExitCode(CommonUtils.max(Duration.ofSeconds(5), remainingTimeout));
        };
        while (true) {
            try (Connection conn = ClickHouseUtils.connect(getHttpHostPort(), timeout);
                 Statement statement = conn.createStatement()) {
                statement.execute("SELECT 1");
                return;
            } catch (SQLRecoverableException | ClickHouseUnknownException e) {
                handleError.accept(e);
            } catch (SQLException e) {
                throw new Checked.CheckedException(e);
            } catch (RuntimeException e) {
                // clickhouse-jdbc может бросить ClickHouseUnknownException, а может
                // сначала завернуть её в строку, потом в RuntimeException и потом бросить
                if (e.getMessage().contains("ClickHouseUnknownException")) {
                    handleError.accept(e);
                } else {
                    throw e;
                }
            }
        }
    }

    public static class Builder {
        private boolean softStop;
        private DockerRunner dockerRunner;
        private Integer hostHttpPort;
        private Integer hostNativePort;

        private Builder() {
            softStop = false;
        }

        public Builder withSoftStop(boolean softStop) {
            this.softStop = softStop;
            return this;
        }

        public Builder withDockerRunner(DockerRunner dockerRunner) {
            this.dockerRunner = dockerRunner.copy();
            return this;
        }

        public Builder withDocker(Docker docker) {
            return withDockerRunner(new DockerRunner(docker, CLICKHOUSE_IMAGE));
        }

        public Builder withHostHttpPort(Integer hostHttpPort) {
            this.hostHttpPort = hostHttpPort;
            return this;
        }

        public Builder withHostNativePort(Integer hostNativePort) {
            this.hostNativePort = hostNativePort;
            return this;
        }

        public ClickHouseContainer build() throws InterruptedException {
            Objects.requireNonNull(dockerRunner);
            if (hostHttpPort != null) {
                dockerRunner.withPublishedPort(hostHttpPort, 8123);
            } else {
                dockerRunner.withPublishedPort(8123);
            }
            if (hostNativePort != null) {
                dockerRunner.withPublishedPort(hostNativePort, 9000);
            } else {
                dockerRunner.withPublishedPort(9000);
            }
            return new ClickHouseContainer(dockerRunner, softStop);
        }
    }
}
