package ru.yandex.direct.clickhouse;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;

import ru.yandex.clickhouse.ClickHouseConnection;
import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.clickhouse.settings.ClickHouseProperties;
import ru.yandex.direct.utils.HostPort;
import ru.yandex.direct.utils.MySQLQuote;

import static ru.yandex.direct.utils.CommonUtils.nvl;

public class ClickHouseUtils {
    public static final Duration CONNECT_TIMEOUT = Duration.ofSeconds(10);

    // Сервер рвет keep-alive соединение раньше чем мы, из-за этого прилетают исключения и сервис
    // останавливается. Поэтому работаем без keep-alive. Можно было бы выставить timeout в ноль,
    // но почему то с нулем происходит то же самое, так что просто делаем таймаут меньше чем на сервере
    // (по умолчанию 3 секунды).
    public static final int KEEP_ALIVE_TIMEOUT_MILLIS = 1;
    public static final long MAX_MEMORY_USAGE = 20_000_000_000L;

    private ClickHouseUtils() {
    }

    /*
     * Можно было бы обернуть SQLException в Checked.CheckedException, но это бесполезно,
     * потому что ClickHouseConnection - это AutoCloseable, у которого close бросает SQLException.
     * Так что полностью избавиться от checked-исключений в этом месте все равно нельзя.
     */
    public static ClickHouseConnection connect(String jdbcUrl, Duration connectTimeout) throws SQLException {
        ClickHouseProperties clickHouseProperties = new ClickHouseProperties();
        clickHouseProperties.setConnectionTimeout(Math.toIntExact(nvl(connectTimeout, CONNECT_TIMEOUT).toMillis()));
        clickHouseProperties.setKeepAliveTimeout(KEEP_ALIVE_TIMEOUT_MILLIS);
        clickHouseProperties.setMaxMemoryUsage(MAX_MEMORY_USAGE);
        return (ClickHouseConnection) new ClickHouseDataSource(
                jdbcUrl,
                clickHouseProperties
        ).getConnection();
    }

    public static ClickHouseConnection connect(HostPort hostPort) throws SQLException {
        return connect(hostPort, CONNECT_TIMEOUT);
    }

    public static ClickHouseConnection connect(HostPort hostPort, Duration connectTimeout) throws SQLException {
        return connect("jdbc:clickhouse://" + hostPort.getHost() + ":" + hostPort.getPort(), connectTimeout);
    }

    public static String quote(String identifier) {
        if (identifier.contains("'") || identifier.contains("\\")) {
            throw new IllegalArgumentException(
                    "zookeeperPath must not contain single quotes or back-slashes: " + identifier);
        }
        return "'" + identifier + "'";
    }

    public static String quoteName(String value) {
        return MySQLQuote.quoteName(value);
    }

    public static String makeSchemaCreationScript(
            Collection<String> hosts,
            String dbName,
            Collection<TableSchema> schemas
    ) {
        StringBuilder script = new StringBuilder();

        script.append(scriptHeader(hosts, dbName));
        for (TableSchema schema : schemas) {
            script.append(queryScript(schema.toString()));
        }

        return script.toString();
    }

    public static String makeDistributedSchemaCreationScript(
            Collection<String> hosts,
            String dbName,
            Collection<TableSchema> schemas,
            String clusterName,
            String shardingKey
    ) {
        String replicatedSuffix = "_replicated";
        StringBuilder script = new StringBuilder();

        script.append(scriptHeader(hosts, dbName));
        for (TableSchema schema : schemas) {
            script.append(queryScript(schema
                    .renamed(schema.getTableName())
                    .toDistributed(dbName, schema.getTableName() + replicatedSuffix, clusterName, shardingKey)
                    .toString()
            ));
            script.append(queryScript(schema
                    .renamed(schema.getTableName() + replicatedSuffix)
                    .toReplicated("/clickhouse/tables/{shard}/" + schema.getTableName() + replicatedSuffix, "{replica}")
                    .toString()
            ));
        }

        return script.toString();
    }

    private static String queryScript(String query) {
        return "for ch in $CLICKHOUSES; do\n" +
                "clickhouse-client -h \"$ch\" <<'END_OF_QUERY'\n" +
                query + "\n" +
                "END_OF_QUERY\n" +
                "done\n";
    }

    private static StringBuilder scriptHeader(Collection<String> hosts, String dbName) {
        StringBuilder script = new StringBuilder();

        script.append("#!/bin/bash\n");
        script.append("set -ex\n");
        script.append("CLICKHOUSES=");
        script.append(String.join(" ", hosts));
        script.append("\n");
        script.append(queryScript("CREATE DATABASE IF NOT EXISTS " + quoteName(dbName)));

        return script;
    }

    /**
     * Получить все партиции в таблице с движком *MergeTree.
     *
     * @param connection Соединение с clickhouse
     * @param dbName     Название БД
     * @param tableName  Название таблицы
     * @return Список имён партиций. Представляет собой даты в формате {@code YYYYMM}.
     */
    public static Collection<String> getTablePartitions(Connection connection, String dbName, String tableName)
            throws SQLException {
        Collection<String> partitions = new ArrayList<>();
        String partitionsSql = "SELECT `partition` FROM `system`.`parts`"
                + " WHERE `database` = ? AND `table` = ? GROUP BY `partition`";
        try (PreparedStatement statement = connection.prepareStatement(partitionsSql)) {
            statement.setString(1, dbName);
            statement.setString(2, tableName);
            try (ResultSet resultSet = statement.executeQuery()) {
                while (resultSet.next()) {
                    partitions.add(resultSet.getString(1));
                }
            }
        }
        return partitions;
    }

    /**
     * Удалить все данные в таблице с движком *MergeTree.
     * При возникновении ошибки во время работы может стереть не все строки.
     *
     * @param connection Соединение с clickhouse
     * @param dbName     Название БД
     * @param tableName  Название таблицы
     * @throws NotALeaderException Этот запрос можно отправлять только на ведущую реплику. Если {@code connection}
     *                             посылает запросы не в ведущую реплику, то удалить данные не получится.
     */
    public static void truncateTable(Connection connection, String dbName, String tableName) throws SQLException {
        Collection<String> partitions = getTablePartitions(connection, dbName, tableName);
        String sql = String.format("ALTER TABLE %s.%s DROP PARTITION ?",
                quoteName(dbName), quoteName(tableName));
        for (String partition : partitions) {
            try (PreparedStatement statement = connection.prepareStatement(sql)) {
                statement.setString(1, partition);
                statement.execute();
            } catch (SQLException e) {
                if (e.getMessage().contains("could be called only on leader replica")) {
                    throw new NotALeaderException(e);
                } else {
                    throw e;
                }
            }
        }
    }

    /**
     * Максимально уменьшить количество партиций в таблице с движком *MergeTree.
     * При возникновении ошибки во время работы может оптимизировать не полностью.
     *
     * @param connection Соединение с clickhouse
     * @param dbName     Название БД
     * @param tableName  Название таблицы
     * @throws NotALeaderException Этот запрос можно отправлять только на ведущую реплику. Если {@code connection}
     *                             посылает запросы не в ведущую реплику, то оптимизировать партиции не получится.
     */
    public static void optimizeTable(Connection connection, String dbName, String tableName) throws SQLException {
        Collection<String> partitions = getTablePartitions(connection, dbName, tableName);
        String sqlTemplate = String.format("OPTIMIZE TABLE %s.%s PARTITION %%s FINAL",
                quoteName(dbName), quoteName(tableName));
        for (String partition : partitions) {
            try (Statement statement = connection.createStatement()) {
                statement.executeUpdate(String.format(sqlTemplate, partition));
            } catch (SQLException e) {
                if (e.getMessage().contains("could be called only on leader replica")) {
                    throw new NotALeaderException(e);
                } else {
                    throw e;
                }
            }
        }
    }
}
