package ru.yandex.direct.clickhouse;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import ru.yandex.direct.utils.Checked;

public class InsertRecord {
    private List<ClickHouseInsertable<?>> fields;
    private PreparedStatement statement;
    private int fieldNo;

    public InsertRecord(List<ClickHouseInsertable<?>> fields, PreparedStatement statement) {
        this.fields = fields;
        this.statement = statement;
        this.fieldNo = 1;
    }

    public <T> InsertRecord setNext(ClickHouseInsertable<T> field, T value) {
        if (field != fields.get(fieldNo - 1)) {
            throw new IllegalStateException("Expected field " + fields.get(fieldNo - 1) + ", got " + field);
        }
        field.getType().setNextInRecord(this, value);
        return this;
    }

    public <T> void setNext(ClickHouseType<T> type, T value) {
        type.setNextInRecord(this, value);
    }

    public void setNext(long value) {
        invokeSetter(() -> statement.setLong(fieldNo, value));
    }

    public void setNext(int value) {
        invokeSetter(() -> statement.setInt(fieldNo, value));
    }

    public void setNext(float value) {
        invokeSetter(() -> statement.setFloat(fieldNo, value));
    }

    public void setNext(double value) {
        invokeSetter(() -> statement.setDouble(fieldNo, value));
    }

    public void setNext(String value) {
        invokeSetter(() -> statement.setString(fieldNo, value));
    }

    public void setNext(LocalDate value) {
        invokeSetter(() -> statement.setDate(fieldNo, Date.valueOf(value)));
    }

    public void setNext(LocalDateTime value) {
        invokeSetter(() -> statement.setTimestamp(fieldNo, Timestamp.valueOf(value)));
    }

    public <T> void setNext(Collection<T> value) {
        invokeSetter(() -> statement.setObject(fieldNo, value));
    }

    private void invokeSetter(Checked.CheckedRunnable<SQLException> setter) {
        try {
            setter.run();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
        fieldNo += 1;
    }
}
