package ru.yandex.direct.clickhouse;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.utils.Checked;

public class InsertStatement implements AutoCloseable {
    private InsertRecord currentRow;
    private final PreparedStatement statement;
    private final List<ClickHouseInsertable<?>> fields;
    private int rowsCount;
    private boolean executed;

    public InsertStatement(ClickHouseTable table, List<ClickHouseInsertable<?>> fields) {
        this.currentRow = null;
        this.fields = fields;
        this.rowsCount = 0;
        this.executed = false;
        try {
            // Преобразование в ClickHousePreparedStatement нужно для обхода баги в jdbc клиенте кликхауса
            this.statement = table.getConn().prepareStatement(
                    "INSERT INTO " + table.getQuotedFullTableName() + " (" +
                            fields.stream().map(ClickHouseSelectable::getExpr).collect(Collectors.joining(", ")) +
                            ") VALUES (" + fields.stream().map(x -> "?").collect(Collectors.joining(", ")) + ")"
            );
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    public int getRowsCount() {
        return rowsCount;
    }

    public InsertRecord newRow() {
        if (executed) {
            throw new IllegalStateException("Can't add rows to executed InsertStatement");
        }
        flushRow();
        currentRow = new InsertRecord(fields, statement);
        return currentRow;
    }

    public void execute() {
        if (executed) {
            throw new IllegalStateException("This InsertStatement is already executed");
        }
        executed = true;
        flushRow();
        long insertedRows = 0L;
        if (rowsCount > 0) {
            try {
                for (int count : statement.executeBatch()) {
                    insertedRows += count;
                }
            } catch (SQLException exc) {
                throw new Checked.CheckedException(exc);
            }
            if (insertedRows != rowsCount) {
                throw new IllegalStateException(
                        "Inserted number of rows (" + insertedRows + ") differs from expected (" + rowsCount + ")"
                );
            }
        }
    }

    private void flushRow() {
        if (currentRow != null) {
            try {
                statement.addBatch();
            } catch (SQLException exc) {
                throw new Checked.CheckedException(exc);
            }
            rowsCount += 1;
            currentRow = null;
        }
    }

    @Override
    public void close() {
        try {
            statement.close();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }
}
