package ru.yandex.direct.clickhouse;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.clickhouse.types.ArrayClickHouseType;
import ru.yandex.direct.utils.Checked;

public class NestedField implements ClickHouseField {
    private final String name;
    private final List<SimpleField<?>> fields;

    public NestedField(String name) {
        this.name = name;
        this.fields = new ArrayList<>();
    }

    public <T> SubFieldArray<T> add(String subFieldName, ClickHouseType<T> type, TypeReference<List<T>> typeReference) {
        fields.add(new SimpleField<>(subFieldName, type));
        return new SubFieldArray<>(name, subFieldName, new ArrayClickHouseType<>(type, typeReference));
    }

    @Override
    public String toSchemaString() {
        return name + " Nested(" +
                fields.stream().map(SimpleField::toSchemaString).collect(Collectors.joining(", ")) +
                ")";
    }

    public static class SubFieldArray<T> implements ClickHouseInsertable<List<T>> {
        private final String parentName;
        private final String name;
        private final ArrayClickHouseType<T> type;

        public SubFieldArray(String parentName, String name, ArrayClickHouseType<T> type) {
            this.parentName = parentName;
            this.name = name;
            this.type = type;
        }

        @Override
        public ClickHouseType<List<T>> getType() {
            return type;
        }

        @Override
        public List<T> from(ResultSet resultSet) {
            return type.fromJavaSqlResult(Checked.get(() -> resultSet.getArray(parentName + "." + name).getArray()));
        }

        @Override
        public String getExpr() {
            return ClickHouseUtils.quoteName(parentName) + "." + ClickHouseUtils.quoteName(name);
        }

        @Override
        public String toString() {
            return "SubFieldArray{parentName='" + parentName + "', name='" + name +
                    "', type='" + type.getTypeExpr() + "'}";
        }
    }
}
