package ru.yandex.direct.clickhouse;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import one.util.streamex.StreamEx;

import ru.yandex.clickhouse.ClickHouseUtil;

public class SqlBuilder {
    private List<String> select = new ArrayList<>();
    private boolean distinct = false;
    private String comment = null;
    private String dbName = null;
    private String from = null;
    private SqlBuilder fromNestedQuery = null;
    private String arrayJoin = null;
    private List<String> join = new ArrayList<>();
    private List<String> prewhere = new ArrayList<>();
    private List<Object> prewhereBindings = new ArrayList<>();
    private List<String> where = new ArrayList<>();
    private List<Object> whereBindings = new ArrayList<>();
    private List<String> orderBy = new ArrayList<>();
    private List<String> groupBy = new ArrayList<>();
    private List<String> having = new ArrayList<>();
    private List<Object> havingBindings = new ArrayList<>();
    private StringBuilder tail = new StringBuilder();
    private List<Object> tailBindings = new ArrayList<>();
    private boolean withTotals = false;
    private boolean selectFinal = false;

    public static class Column {
        protected String name;

        public Column(String name) {
            this.name = name;
        }

        @Override
        public String toString() {
            return ClickHouseUtil.quoteIdentifier(name);
        }
    }

    public static Column column(String name) {
        return new Column(name);
    }

    public enum Order {
        ASC,
        DESC,
    }

    public SqlBuilder select(Collection<String> columns) {
        return select(columns.stream()
                .map(SqlBuilder::column)
                .collect(Collectors.toList())
                .toArray(new Column[0]));
    }

    public SqlBuilder select(String... columns) {
        return select(Arrays.asList(columns));
    }

    public SqlBuilder select(Column... columns) {
        for (Column column : columns) {
            select.add(column.toString());
        }
        return this;
    }

    public SqlBuilder selectExpression(String expression, String alias, boolean isConstantValue) {
        var expr = expression != null && isConstantValue ? "'" + expression + "'" :
                expression;
        select.add(expr + " as " + ClickHouseUtil.quoteIdentifier(alias));
        return this;
    }

    public SqlBuilder selectExpression(String expression, String alias) {
        return selectExpression(expression, alias, false);
    }

    public SqlBuilder selectExpression(ExpressionWithAlias expressionWithAlias) {
        return selectExpression(expressionWithAlias.getExpression(), expressionWithAlias.getAlias());
    }

    public SqlBuilder select(ClickHouseSelectable... columns) {
        for (ClickHouseSelectable column : columns) {
            select.add(column.getExpr());
        }
        return this;
    }

    public SqlBuilder distinct() {
        this.distinct = true;
        return this;
    }

    public SqlBuilder withComment(String comment) {
        if (!comment.startsWith("/*") || !comment.endsWith("*/")) {
            throw new IllegalArgumentException("Comment must be surrounded with /* */");
        }
        this.comment = comment;
        return this;
    }

    public SqlBuilder from(String dbName, String table) {
        this.dbName = dbName;
        from = table;
        fromNestedQuery = null;
        return this;
    }

    public SqlBuilder from(String table) {
        from = table;
        fromNestedQuery = null;
        return this;
    }

    public SqlBuilder from(SqlBuilder nestedQuery){
        from = null;
        fromNestedQuery = nestedQuery;
        return this;
    }

    public SqlBuilder selectFinal(boolean flag) {
        selectFinal = flag;
        return this;
    }

    public SqlBuilder selectFinal() {
        return selectFinal(true);
    }

    public SqlBuilder arrayJoin(String col) {
        arrayJoin = col;
        return this;
    }

    public SqlBuilder join(String expression) {
        return join("INNER JOIN", expression);
    }

    public SqlBuilder join(String joinType, String expression) {
        join.add(joinType + " " + expression);
        return this;
    }

    public SqlBuilder where(String expression, Object... values) {
        where.add(expression);
        whereBindings.addAll(Arrays.asList(values));
        return this;
    }

    public SqlBuilder where(ExpressionWithBinds expressionWithBinds) {
        return where(expressionWithBinds.getExpression(), expressionWithBinds.getBinds());
    }

    public SqlBuilder where(String expression, List<Object> values) {
        // без этого метода where(xxx, asList()) работает совсем не как предполагается
        return where(expression, values.toArray());
    }

    public SqlBuilder where(ClickHouseSelectable column, String op, Object value) {
        where.add(column.getExpr() + " " + op + " ?");
        whereBindings.add(value);
        return this;
    }

    public SqlBuilder where(Column column, String op, Object value) {
        where.add(column.toString() + " " + op + " ?");
        whereBindings.add(value);
        return this;
    }

    public SqlBuilder whereNot(ExpressionWithBinds expressionWithBinds) {
        return where("NOT (" + expressionWithBinds.getExpression() + ")", expressionWithBinds.getBinds());
    }

    public SqlBuilder whereIn(Column column, Collection<?> vals) {
        ExpressionWithBinds e = whereInExpression(column, vals);
        return where(e.expression, e.binds);
    }

    public SqlBuilder whereNotIn(Column column, Collection<?> vals) {
        ExpressionWithBinds e = whereNotInExpression(column, vals);
        return where(e.expression, e.binds);
    }

    private ExpressionWithBinds whereInExpression(Column column, Collection<?> vals) {
        if (vals.isEmpty()) {
            return new ExpressionWithBinds("0");
        } else if (vals.size() == 1) {
            return new ExpressionWithBinds(column + " = ?", vals.iterator().next());
        } else {
            String expr = column.toString() + " in ("
                    + StreamEx.of(vals).map(x -> "?").joining(", ")
                    + ")";
            return new ExpressionWithBinds(expr, vals.toArray());
        }
    }

    private ExpressionWithBinds whereNotInExpression(Column column, Collection<?> vals) {
        if (vals.isEmpty()) {
            return new ExpressionWithBinds("0");
        } else if (vals.size() == 1) {
            return new ExpressionWithBinds(column + " != ?", vals.iterator().next());
        } else {
            String expr = column.toString() + " not in ("
                    + StreamEx.of(vals).map(x -> "?").joining(", ")
                    + ")";
            return new ExpressionWithBinds(expr, vals.toArray());
        }
    }

    public static ExpressionWithBinds joinExpressions(
            String delimiter,
            List<ExpressionWithBinds> expressions
    ) {
        Preconditions.checkArgument(!expressions.isEmpty(), "empty list of expressions");

        if (expressions.size() == 1) {
            return expressions.get(0);
        }

        return new ExpressionWithBinds(
                StreamEx.of(expressions).map(e -> e.expression).joining(delimiter, "(", ")"),
                expressions.stream().flatMap(e -> Arrays.stream(e.binds)).toArray()
        );
    }

    public SqlBuilder prewhere(String expression, Object... values) {
        prewhere.add(expression);
        prewhereBindings.addAll(Arrays.asList(values));
        return this;
    }

    public SqlBuilder prewhere(ExpressionWithBinds expressionWithBinds) {
        return prewhere(expressionWithBinds.getExpression(), expressionWithBinds.getBinds());
    }

    public SqlBuilder prewhere(Column column, String op, Object value) {
        prewhere.add(column.toString() + " " + op + " ?");
        prewhereBindings.add(value);
        return this;
    }

    public SqlBuilder groupBy(ClickHouseSelectable... columns) {
        for (ClickHouseSelectable column : columns) {
            groupBy.add(column.getExpr());
        }
        return this;
    }

    public SqlBuilder groupBy(String... columns) {
        for (String column : columns) {
            groupBy.add(ClickHouseUtil.quoteIdentifier(column));
        }
        return this;
    }

    public SqlBuilder groupByExpression(String... expressions) {
        groupBy.addAll(Arrays.asList(expressions));
        return this;
    }

    public SqlBuilder withTotals() {
        this.withTotals = true;
        return this;
    }

    public SqlBuilder having(String expression, Object... values) {
        having.add(expression);
        havingBindings.addAll(Arrays.asList(values));
        return this;
    }

    public SqlBuilder having(ExpressionWithBinds expressionWithBinds) {
        return having(expressionWithBinds.getExpression(), expressionWithBinds.getBinds());
    }

    public SqlBuilder having(Column column, String op, Object value) {
        having.add(column.toString() + " " + op + " ?");
        havingBindings.add(value);
        return this;
    }

    public SqlBuilder orderBy(String column, Order sortOrder) {
        return orderBy(new Column(column), sortOrder);
    }

    public SqlBuilder orderBy(Column column, Order sortOrder) {
        return orderBy(Collections.singletonList(column), sortOrder);
    }

    public SqlBuilder orderBy(List<Column> columns, Order sortOrder) {
        for (Column col : columns) {
            orderBy.add(col.toString() + " " + sortOrder.toString());
        }
        return this;
    }

    public SqlBuilder orderByExpression(String expression, Order sortOrder) {
        orderBy.add(expression + " " + sortOrder.name());
        return this;
    }

    public SqlBuilder limit(int limit) {
        tail.append("LIMIT ?");
        tailBindings.add(limit);
        return this;
    }

    public SqlBuilder limit(int offset, int limit) {
        tail.append("LIMIT ?, ?");
        tailBindings.add(offset);
        tailBindings.add(limit);
        return this;
    }

    @Override
    public String toString() {
        return generateSql(false);
    }

    public String generateSql(boolean pretty) {
        StringBuilder sb = new StringBuilder();

        String del = pretty ? "\n" : " ";

        sb.append("SELECT ");
        if (comment != null) {
            sb.append(' ');
            sb.append(comment);
            sb.append(' ');
        }
        if (distinct) {
            sb.append("DISTINCT ");
        }
        sb.append(String.join(", ", select));
        if (from != null) {
            sb.append(del).append("FROM ");
            if (dbName != null) {
                sb.append(ClickHouseUtil.quoteIdentifier(dbName));
                sb.append('.');
            }
            sb.append(ClickHouseUtil.quoteIdentifier(from));
        }
        if (fromNestedQuery != null){
            sb.append(del).append("FROM ").append(del).append("( ");
            sb.append(fromNestedQuery.generateSql(pretty));
            sb.append(del).append(" )");
        }
        if (selectFinal) {
            sb.append(" FINAL");
        }
        if (arrayJoin != null) {
            sb.append(del).append("ARRAY JOIN ");
            sb.append(ClickHouseUtil.quoteIdentifier(arrayJoin));
        }
        if (!join.isEmpty()) {
            sb.append(del).append(String.join(del, join));
        }
        if (!prewhere.isEmpty()) {
            sb.append(del).append("PREWHERE ");
            sb.append(String.join(del + "AND ", prewhere));
        }
        if (!where.isEmpty()) {
            sb.append(del).append("WHERE ");
            sb.append(String.join(del + "AND ", where));
        }
        if (!groupBy.isEmpty()) {
            sb.append(del).append("GROUP BY ");
            sb.append(String.join(", ", groupBy));
            if (withTotals) {
                sb.append(" WITH TOTALS ");
            }
        }
        if (!having.isEmpty()) {
            sb.append(del).append("HAVING ");
            sb.append(String.join(del + "AND ", having));
        }
        if (!orderBy.isEmpty()) {
            sb.append(del).append("ORDER BY ");
            sb.append(String.join(", ", orderBy));
        }
        if (tail.length() > 0) {
            sb.append(del).append(tail);
        }
        return sb.toString();
    }

    public Object[] getBindings() {
        List<Object> bindings = new ArrayList<>();
        if (fromNestedQuery != null){
            bindings.addAll(Arrays.asList(fromNestedQuery.getBindings()));
        }

        bindings.addAll(prewhereBindings);
        bindings.addAll(whereBindings);
        bindings.addAll(havingBindings);
        bindings.addAll(tailBindings);
        return bindings.toArray();
    }

    public static class ExpressionWithBinds {
        private final String expression;
        private final Object[] binds;

        public ExpressionWithBinds(String expression, Object... binds) {
            this.expression = expression;
            this.binds = binds;
        }

        public ExpressionWithBinds(String expression, List<Object> binds) {
            this.expression = expression;
            this.binds = binds.toArray();
        }

        public String getExpression() {
            return expression;
        }

        public Object[] getBinds() {
            return binds;
        }
    }

    public static class ExpressionWithAlias {
        private final String expression;
        private final String alias;

        public ExpressionWithAlias(String expression, String alias) {
            this.expression = expression;
            this.alias = alias;
        }

        public String getExpression() {
            return expression;
        }

        public String getAlias() {
            return alias;
        }
    }
}
