package ru.yandex.direct.clickhouse;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class TableSchema {
    private String dbName;
    private String tableName;
    private List<ClickHouseField> fields;
    private String engine;
    private List<String> engineArgs;

    public TableSchema(String dbName, String tableName, List<ClickHouseField> fields,
                       String engine, List<String> engineArgs) {
        this.dbName = dbName;
        this.tableName = tableName;
        this.fields = fields;
        this.engine = engine;
        this.engineArgs = engineArgs;
    }

    public String getDbName() {
        return dbName;
    }

    public String getTableName() {
        return tableName;
    }

    public String getQuotedFullTableName() {
        return ClickHouseUtils.quoteName(dbName) + "." + ClickHouseUtils.quoteName(tableName);
    }

    public void createTable(Connection conn) throws SQLException {
        createTable(conn, false);
    }

    public void createTable(Connection conn, boolean ifNotExists) throws SQLException {
        try (PreparedStatement stmt = conn.prepareStatement(this.toCreateTable(ifNotExists))) {
            stmt.executeUpdate();
        }
    }

    @Override
    public String toString() {
        return toCreateTable(false);
    }

    public String toCreateTable(boolean ifNotExists) {
        return "CREATE TABLE " + (ifNotExists ? "IF NOT EXISTS " : "") +
                ClickHouseUtils.quoteName(dbName) + "." + ClickHouseUtils.quoteName(tableName) + " ON CLUSTER logs (\n" +
                fields.stream().map(field -> "    " + field.toSchemaString()).collect(Collectors.joining(",\n")) +
                "\n) ENGINE = " + engine + "(\n" +
                engineArgs.stream().map(arg -> "    " + arg).collect(Collectors.joining(",\n")) +
                "\n)";
    }

    public TableSchema toReplicated(String zookeeperPath, String replicaName) {
        if (!engine.endsWith("MergeTree") || engine.startsWith("Replicated")) {
            throw new IllegalStateException("Engine " + engine + " can't be replicated");
        } else {
            return new TableSchema(
                    dbName,
                    tableName,
                    fields,
                    "Replicated" + engine,
                    Stream.concat(
                            Stream.of(ClickHouseUtils.quote(zookeeperPath), ClickHouseUtils.quote(replicaName)),
                            engineArgs.stream()).collect(Collectors.toList()
                    )
            );
        }
    }

    public TableSchema toDistributed(String dbName, String tableName, String cluster, String shardingKey) {
        return new TableSchema(
                this.dbName,
                this.tableName,
                this.fields,
                "Distributed",
                Arrays.asList(
                        ClickHouseUtils.quoteName(cluster),
                        ClickHouseUtils.quoteName(dbName),
                        ClickHouseUtils.quoteName(tableName),
                        shardingKey
                )
        );
    }

    public TableSchema renamed(String newTableName) {
        return new TableSchema(
                dbName,
                newTableName,
                fields,
                engine,
                engineArgs
        );
    }

}
