package ru.yandex.direct.clickhouse.types;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.direct.clickhouse.ClickHouseType;
import ru.yandex.direct.clickhouse.InsertRecord;

public class DateTimeClickHouseType extends ClickHouseType<LocalDateTime> {
    public DateTimeClickHouseType() {
        super("DateTime");
    }

    @Override
    public LocalDateTime fromClickHouseString(@Nonnull String stringValue) {
        return LocalDateTime.parse(
                stringValue,
                DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")
        );
    }

    @Nullable
    @Override
    public LocalDateTime fromJavaSqlResult(@Nullable Object value) {
        if (value == null) {
            return null;
        }
        // Нельзя использовать Timestamp::toLocalDateTime, т.к. она автоматически преобразует дату в текущую тайм-зону
        // Во всех остальных местах даты трактуются как UTC.
        return LocalDateTime.ofInstant(((Timestamp) value).toInstant(), ZoneId.of("UTC"));
    }

    @Override
    public void setNextInRecord(InsertRecord record, LocalDateTime value) {
        record.setNext(value);
    }

    @Override
    public Object toSqlObject(LocalDateTime value) {
        return java.sql.Timestamp.from(value.toInstant(ZoneOffset.UTC));
    }
}
