package ru.yandex.direct.clickhouse.types;

import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableBiMap;

import ru.yandex.direct.clickhouse.ClickHouseType;
import ru.yandex.direct.clickhouse.InsertRecord;

/**
 * Сохраняет вместо какого-то константного объекта какой-то номер. При чтении какого-то номера возвращает какой-то
 * константный объект. По сути это Enum-тип, который хранится в кликхаусе как обычное число.
 */
@ParametersAreNonnullByDefault
public class MappingClickHouseType<E> extends ClickHouseType<E> {
    private final Map<E, Integer> serializeMap;
    private final Map<Integer, E> deserializeMap;

    public MappingClickHouseType(String type, ImmutableBiMap<E, Integer> enumMap) {
        super(type);
        this.serializeMap = enumMap;
        this.deserializeMap = enumMap.inverse();
    }

    @Override
    public E fromClickHouseString(String stringValue) {
        return Objects.requireNonNull(deserializeMap.get(Integer.parseInt(stringValue)));
    }

    @Override
    public E fromJavaSqlResult(@Nullable Object value) {
        return Objects.requireNonNull(deserializeMap.get((Integer) value));
    }

    @Override
    public void setNextInRecord(InsertRecord record, E value) {
        record.setNext(Objects.requireNonNull(serializeMap.get(value)));
    }

    @Override
    public Object toSqlObject(E value) {
        return Objects.requireNonNull(serializeMap.get(value));
    }
}
