package ru.yandex.direct.clickhouse.types;

import java.util.OptionalLong;

import ru.yandex.direct.clickhouse.ClickHouseType;
import ru.yandex.direct.clickhouse.InsertRecord;

public class OptionalLongClickHouseType extends ClickHouseType<OptionalLong> {
    private final long emptyValue;
    private final boolean allowIllegalValue;

    // allowIllegalValue сделан для случая нулевых reqid, которых обычно не бывает и которые мы согласны смешивать
    // с null'ами в случае ошибки (например, DIRECT-95746). Обычно нужно пользоваться конструктором с двумя параметрами
    public OptionalLongClickHouseType(String type, long emptyValue, boolean allowIllegalValue) {
        super(type);
        this.emptyValue = emptyValue;
        this.allowIllegalValue = allowIllegalValue;
    }

    public OptionalLongClickHouseType(String type, long emptyValue) {
        this(type, emptyValue, false);
    }

    @Override
    public void setNextInRecord(InsertRecord record, OptionalLong value) {
        if (value.isPresent()) {
            if (emptyValue == value.getAsLong() && !allowIllegalValue) {
                throw new IllegalArgumentException(
                        "Value " + value + " is illegal because it represents empty value in clickhouse table."
                );
            }
            record.setNext(value.getAsLong());
        } else {
            record.setNext(emptyValue);
        }
    }

    @Override
    public OptionalLong fromClickHouseString(String stringValue) {
        long value = Long.parseLong(stringValue);
        return emptyValue == value ? OptionalLong.empty() : OptionalLong.of(value);
    }

    @Override
    public OptionalLong fromJavaSqlResult(Object value) {
        if (value == null || (Long) value == emptyValue) {
            return OptionalLong.empty();
        } else {
            return OptionalLong.of((Long) value);
        }
    }

    @Override
    public Object toSqlObject(OptionalLong value) {
        return value.orElse(emptyValue);
    }
}
