package ru.yandex.direct.cloud.iam;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Абстрактный класс для получения iam-токенов из Яндекс.Облака в обмен на OAuth-токен аккаунта на Яндексе.
 * Подробнее про iam-токены облака тут:
 * @see <a href="https://docs.yandex-team.ru/cloud/iam/operations/iam-token/create">
 *     Получение IAM-токена для аккаунта на Яндексе</a>
 * Внутри себя при первом обращении за токеном создает задачу в шедулере {@link ISchedulerWithFixedDelay},
 * которая запускается раз в {@value CHECK_IAM_TOKEN_EXPIRE_TIME_INTERVAL_IN_MS} миллисекунд и проверяет
 * не пора ли обновить iam-токен, и если пора, то обновляет его.
 * Так же iam-токен может быть инвалидирован в любой момент методом {@link #invalidateCurrentIamToken()}
 * (например, если поменялся OAuth токен).
 * Внутренний процесс получения iam-токенов запускается при первом обращении к токену,
 * и останавливается методом {@link #stop()}. Этот же метод нужно вызвать при завершении работы с объектом.
 * Протокол получения iam-токена определяют наследники этого класса.
 */
public abstract class AbstractIamTokenProvider implements IIamTokenProvider {
    private static final Logger logger = LoggerFactory.getLogger(AbstractIamTokenProvider.class);
    private static final int CHECK_IAM_TOKEN_EXPIRE_TIME_INTERVAL_IN_MS = 5000;

    private final ISchedulerWithFixedDelay scheduler;
    private ScheduledFuture<?> mainTaskFuture;
    private final IOauthTokenProvider oAuthTokenProvider;

    private boolean stopped;
    private volatile String currentIamToken;
    private volatile Instant currentIamTokenExpireTime;

    private final Object locker;

    private final CloudApiConnectionInfo connectionInfo;

    private final int maxIamTokenTimeoutInMs;

    protected abstract IamTokenInfo getNewIamToken(
            CloudApiConnectionInfo connectionInfo, IOauthTokenProvider oAuthTokenProvider)
            throws IOException, InterruptedException;

    protected abstract void initTransport(CloudApiConnectionInfo connectionInfo);

    protected abstract void closeTransport() throws IOException;

    public AbstractIamTokenProvider(
            ISchedulerWithFixedDelay scheduler,
            CloudApiConnectionInfo connectionInfo,
            IOauthTokenProvider oAuthTokenProvider,
            int createNewTokenIntervalInSeconds) {
        this.scheduler = scheduler;
        this.oAuthTokenProvider = oAuthTokenProvider;
        this.connectionInfo = connectionInfo;
        locker = new Object();
        maxIamTokenTimeoutInMs = (int)Duration.ofSeconds(createNewTokenIntervalInSeconds).toMillis();
        currentIamTokenExpireTime = Instant.now();
        currentIamToken = "";
        stopped = true;
    }

    /**
     * Запускает процесс получения iam-токенов, если он был остановлен
     */
    private void init() {
        synchronized (locker) {
            int initialDelay = 0;
            if (!stopped) {
                return;
            }

            try {
                logger.info("Initializing iam-token provider transport");
                initTransport(connectionInfo);
                logger.info("Trying to receive first iam-token");
                IamTokenInfo newToken = getNewIamTokenWithCorrectedExpireTime(Instant.now());
                currentIamToken = newToken.iamToken;
                currentIamTokenExpireTime = newToken.expireTime;
                // Если при инициализации токен получен успешно, то асинхронную задачу
                // по обновлению токена запускаем чуть погодя
                initialDelay = CHECK_IAM_TOKEN_EXPIRE_TIME_INTERVAL_IN_MS;
            } catch (InterruptedException intex) {
                Thread.currentThread().interrupt();
            } catch (IOException ioex) {
                logger.error("Failed to get iam token on initialization", ioex);
            }
            mainTaskFuture = scheduler.scheduleWithFixedDelay(
                    this::updateIamToken,
                    initialDelay,
                    CHECK_IAM_TOKEN_EXPIRE_TIME_INTERVAL_IN_MS,
                    TimeUnit.MILLISECONDS);
            stopped = false;
        }
    }

    private void updateIamToken() {
        try {
            if (checkStopped()) {
                return;
            }
            Instant now = Instant.now();
            Instant tokenExpireTime;
            // Создаем оптимистичную блокировку на инвалидацию токена
            synchronized (locker) {
                tokenExpireTime = currentIamTokenExpireTime;
            }
            logger.debug("It's time to check if the current iam-token has expired");
            Duration duration = Duration.between(now, tokenExpireTime);
            if (!duration.isNegative()) {
                logger.debug(String.format(
                        "The current iam-token is valid for another %.3f seconds, so we won't receive a new one",
                        duration.toMillis() / 1000.0));
                return;
            }
            logger.info("The current iam-token is has expired, so trying to receive a new one");
            IamTokenInfo newToken = getNewIamTokenWithCorrectedExpireTime(now);

            synchronized (locker) {
                // Проверяем оптимистичную блокировку на инвалидацию токена
                if (currentIamTokenExpireTime.equals(tokenExpireTime)) {
                    // если все хорошо, то устанавливаем новый токен и его время жизни
                    currentIamToken = newToken.iamToken;
                    currentIamTokenExpireTime = newToken.expireTime;
                    logger.info("The new iam-token has been successfully received, it is valid until {}",
                            LocalDateTime.ofInstant(currentIamTokenExpireTime, ZoneId.systemDefault()));
                } else {
                    // Если нет, то токен и его время жизни не трогаем, все равно через 5 секунд
                    // при вызове этого же метода по расписанию, будет получен новый токен
                    logger.warn("Token invalidation detected, " +
                                    "most likely due to invalidateCurrentIamToken() method call. Do nothing.");
                }
            }

        } catch (InterruptedException intex) {
            // Ничего не пишем в лог, так как это сигнал об окончании работы
            Thread.currentThread().interrupt();
        } catch (IOException ioex) {
            logger.error("Failed to get iam-token, let's try again a little later", ioex);
        }
    }

    private IamTokenInfo getNewIamTokenWithCorrectedExpireTime(Instant now) throws IOException, InterruptedException {
        IamTokenInfo iamToken = getNewIamToken(connectionInfo, oAuthTokenProvider);
        Instant maxExpireTime = now.plusMillis(maxIamTokenTimeoutInMs);
        Instant tokenExpireTime = iamToken.expireTime;
        if (Duration.between(tokenExpireTime, maxExpireTime).isNegative()) {
            tokenExpireTime = maxExpireTime;
        }
        return new IamTokenInfo(iamToken.iamToken, tokenExpireTime);
    }

    private boolean checkStopped() throws InterruptedException {
        synchronized (locker) {
            return stopped || Thread.currentThread().isInterrupted();
        }
    }

    /**
     * Останавливает процесс получения iam-токенов, если он был запущен
     * @return false, если процесс не был запущен и true, если уже был.
     */
    public boolean stop() {
        synchronized (locker) {
            if (stopped) {
                return false;
            }
            mainTaskFuture.cancel(true);
            try {
                closeTransport();
            } catch (IOException ioex) {
                logger.error("Something goes wrong while closing iam-token provider transport", ioex);
            }
            stopped = true;
            currentIamTokenExpireTime = Instant.MIN;
            currentIamToken = "";
            return true;
        }
    }

    /**
     * Возвращает текущий iam-токен. Если поставщик еще не запущен, инициализирует его
     * @return текущий iam-токен
     */
    @Override
    public String getCurrentIamToken() {
        if (stopped) {
            init();
        }
        return currentIamToken;
    }

    /**
     * Инвалидирует текущий iam-токен. Токен обновится не сразу, а при следующем запуске {@link #updateIamToken()}
     * по расписанию (раз в 5 секунд)
     */
    @Override
    public void invalidateCurrentIamToken() {
        synchronized (locker) {
            // Инвалидируем iam-токен. При следующем вызове метода updateIamToken() по расписанию
            // будет получен новый токен. Если инвалидация застанет метод updateIamToken() в процессе выполнения,
            // то из-за оптимистичной блокировки в методе updateIamToken() токен не будет заменен на невалидный.
            currentIamTokenExpireTime = Instant.now();
        }
    }

    protected static class IamTokenInfo {
        private final String iamToken;
        private final Instant expireTime;

        protected IamTokenInfo(String iamToken, Instant expireTime) {
            this.iamToken = iamToken;
            this.expireTime = expireTime;
        }
    }

}
