package ru.yandex.direct.cloud.iam;

import java.io.IOException;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import io.grpc.Deadline;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.MetadataUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import yandex.cloud.api.iam.v1.IamTokenServiceGrpc;
import yandex.cloud.api.iam.v1.IamTokenServiceOuterClass;

/**
 * Поставщик iam-токенов Яндекс.Облака в обмен на OAuth-токен аккаунта на Яндексе по протоколу gRPC.
 */
public class GrpcIamTokenProvider extends AbstractIamTokenProvider {
    private static final Logger logger = LoggerFactory.getLogger(GrpcIamTokenProvider.class);

    protected ManagedChannel grpcChannel;

    static final Metadata.Key<String> X_CLIENT_REQUEST_ID =
            Metadata.Key.of("x-client-request-id", Metadata.ASCII_STRING_MARSHALLER);
    static final Metadata.Key<String> X_CLIENT_TRACE_ID =
            Metadata.Key.of("x-client-trace-id", Metadata.ASCII_STRING_MARSHALLER);

    public GrpcIamTokenProvider(
            ISchedulerWithFixedDelay scheduler, CloudApiConnectionInfo connectionInfo,
            IOauthTokenProvider oAuthTokenProvider, int createNewTokenIntervalInSeconds) {
        super(scheduler, connectionInfo, oAuthTokenProvider, createNewTokenIntervalInSeconds);
    }

    @Override
    protected void initTransport(CloudApiConnectionInfo connectionInfo) {
        this.grpcChannel = ManagedChannelBuilder
                .forAddress(connectionInfo.getApiUrl(), connectionInfo.getApiPort())
                .userAgent(connectionInfo.getUserAgent())
                .keepAliveWithoutCalls(true)
                .keepAliveTime(15, TimeUnit.SECONDS)
                .keepAliveTimeout(10, TimeUnit.SECONDS)
                .build();
    }

    @Override
    protected IamTokenInfo getNewIamToken(CloudApiConnectionInfo connectionInfo, IOauthTokenProvider oAuthTokenProvider)
            throws IOException {
        Metadata metadata = createMeta();

        IamTokenServiceGrpc.IamTokenServiceBlockingStub stub = IamTokenServiceGrpc.newBlockingStub(grpcChannel)
                .withInterceptors(MetadataUtils.newAttachHeadersInterceptor(metadata));

        IamTokenServiceOuterClass.CreateIamTokenRequest request =
                IamTokenServiceOuterClass.CreateIamTokenRequest.newBuilder()
                        .setYandexPassportOauthToken(oAuthTokenProvider.getCurrentOAuthToken())
                        .build();

        IamTokenServiceOuterClass.CreateIamTokenResponse response;
        String methodInfo = String.format("%s on %s, %s = %s, %s = %s",
                    IamTokenServiceGrpc.getCreateMethod().getFullMethodName(), grpcChannel.authority(),
                    X_CLIENT_REQUEST_ID.name(), metadata.get(X_CLIENT_REQUEST_ID),
                    X_CLIENT_TRACE_ID.name(), metadata.get(X_CLIENT_TRACE_ID));

        long nanos = System.nanoTime();
        try {
            logger.info("Trying to get iam-token by method {}", methodInfo);
            response = stub
                    .withDeadline(Deadline.after(connectionInfo.getRequestTimeoutInMs(), TimeUnit.MILLISECONDS))
                    .create(request);
        } catch (StatusRuntimeException ex) {
            throw new IOException(
                    String.format("Something goes wrong while receiving iam-token by method %s", methodInfo), ex);
        } finally {
            double seconds = (System.nanoTime() - nanos) / 1_000_000_000.0;
            logger.info(String.format("New iam-token received by method %s in %.3f seconds", methodInfo, seconds));
        }
        return fromGrpcResponse(response);
    }

    private IamTokenInfo fromGrpcResponse(IamTokenServiceOuterClass.CreateIamTokenResponse response) {
        Instant tokenExpireTime = Instant.ofEpochSecond(
                response.getExpiresAt().getSeconds(), response.getExpiresAt().getNanos());
        return new IamTokenInfo(response.getIamToken(), tokenExpireTime);
    }

    private Metadata createMeta() {
        Metadata headers = new Metadata();
        headers.put(X_CLIENT_REQUEST_ID, UUID.randomUUID().toString());
        headers.put(X_CLIENT_TRACE_ID, UUID.randomUUID().toString());
        return headers;
    }

    @Override
    public void closeTransport() {
        grpcChannel.shutdown();
    }
}
