package ru.yandex.direct.cloud.mdb.mysql.api.grpc;

import java.io.Closeable;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.cloud.iam.CloudApiConnectionInfo;
import ru.yandex.direct.cloud.iam.IIamTokenProvider;
import ru.yandex.direct.cloud.mdb.mysql.api.BaseCloudApiHelper;

/**
 * Класс-помощник для осуществления grpc запросов к cloud mdb mysql api
 */
public class GrpcCloudApiHelper extends BaseCloudApiHelper implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(GrpcCloudApiHelper.class);

    private final ManagedChannel grpcChannel;

    public static final Metadata.Key<String> AUTHORIZATION =
            Metadata.Key.of("Authorization", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> X_CLIENT_REQUEST_ID =
            Metadata.Key.of("x-client-request-id", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> X_CLIENT_TRACE_ID =
            Metadata.Key.of("x-client-trace-id", Metadata.ASCII_STRING_MARSHALLER);

    public GrpcCloudApiHelper(CloudApiConnectionInfo connectionInfo, IIamTokenProvider iamTokenProvider) {
        super(connectionInfo, iamTokenProvider);
        this.grpcChannel = ManagedChannelBuilder
                .forAddress(connectionInfo.getApiUrl(), connectionInfo.getApiPort())
                .userAgent(connectionInfo.getUserAgent())
                .maxInboundMessageSize(256 * 1024 * 1024)
                .keepAliveWithoutCalls(true)
                .keepAliveTime(15, TimeUnit.SECONDS)
                .keepAliveTimeout(10, TimeUnit.SECONDS)
                .build();
    }

    public Metadata createMeta() {
        String iamToken = getIamTokenProvider().getCurrentIamToken();
        Metadata headers = new Metadata();
        headers.put(AUTHORIZATION, "Bearer " + iamToken);
        headers.put(X_CLIENT_REQUEST_ID, UUID.randomUUID().toString());
        headers.put(X_CLIENT_TRACE_ID, UUID.randomUUID().toString());
        return headers;
    }

    public GrpcCallInfo createCallInfo(String methodName, Metadata metadata, String requestMessage) {
        String methodInfo = String.format("%s/%s(%s), %s = %s, %s = %s",
                grpcChannel.authority(), methodName, formatMessage(requestMessage),
                X_CLIENT_REQUEST_ID.name(), metadata.get(X_CLIENT_REQUEST_ID),
                X_CLIENT_TRACE_ID.name(), metadata.get(X_CLIENT_TRACE_ID));

        return new GrpcCallInfo(methodInfo, System.nanoTime());
    }

    public void logBeginCall(GrpcCallInfo callInfo) {
        logger.info("Trying to call method {}", callInfo.getMethodInfo());
    }

    public void logSuccessEndCall(GrpcCallInfo callInfo) {
        double seconds = (System.nanoTime() - callInfo.getBeginNanoTime()) / 1_000_000_000.0;
        logger.info(String.format("Call to method %s successfully completed in %.3f seconds",
                callInfo.getMethodInfo(), seconds));
    }

    public void logErrorEndCall(GrpcCallInfo callInfo, StatusRuntimeException ex) {
        double seconds = (System.nanoTime() - callInfo.getBeginNanoTime()) / 1_000_000_000.0;
        logger.error(String.format("Call to method %s failed with exception in %.3f seconds",
                callInfo.getMethodInfo(), seconds), ex);
    }

    private static String formatMessage(String message) {
        if (message == null)
            return null;
        char lastNotWhitespaceChar = '{';
        char lastChar = '{';
        StringBuilder sbld = new StringBuilder();
        int pos = 0;
        boolean inQuota = false;
        while (pos < message.length()) {
            char c = message.charAt(pos);
            if (inQuota) {
                if (c == '"' && lastChar != '\\') {
                    inQuota = false;
                } else if (c == '\n') {
                    sbld.append('\\');
                    c = 'n';
                }
                sbld.append(c);
            } else {
                if (!Character.isWhitespace(c)) {
                    if (c == '"') {
                        inQuota = true;
                    }
                    if (c != ',' && c != '}' && lastNotWhitespaceChar != '{' && Character.isWhitespace(lastChar)) {
                        sbld.append(' ');
                    }
                    lastNotWhitespaceChar = c;
                    sbld.append(c);
                } else if (c == '\n' && pos < message.length() - 1) {
                    if (message.charAt(pos + 1) != '}' && lastNotWhitespaceChar != '{') {
                        lastNotWhitespaceChar = ',';
                        sbld.append(',');
                    }
                    c = ' ';
                }
            }
            lastChar = c;
            pos++;
        }
        return sbld.toString();
    }

    public ManagedChannel getGrpcChannel() {
        return grpcChannel;
    }

    @Override
    public void close() {
        grpcChannel.shutdown();
    }
}
