package ru.yandex.direct.cloud.mdb.mysql.api.grpc;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import io.grpc.Deadline;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import yandex.cloud.api.mdb.mysql.v1.ClusterOuterClass;
import yandex.cloud.api.mdb.mysql.v1.ClusterServiceGrpc;
import yandex.cloud.api.mdb.mysql.v1.ClusterServiceOuterClass;

import ru.yandex.direct.cloud.mdb.mysql.api.transport.ClusterRawInfo;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListResponse;

import static io.grpc.stub.MetadataUtils.newAttachHeadersInterceptor;

/**
 * Реализация метода получения списка кластеров облака по протоколу gRPC
 */
class GrpcClustersReader {
    public static GetClustersListResponse getClustersList(GrpcCloudApiHelper helper, GetClustersListRequest request)
            throws IOException {
        Metadata metadata = helper.createMeta();

        ClusterServiceGrpc.ClusterServiceBlockingStub stub = ClusterServiceGrpc.newBlockingStub(helper.getGrpcChannel())
                .withInterceptors(newAttachHeadersInterceptor(metadata));

        ClusterServiceOuterClass.ListClustersRequest.Builder requestBuilder =
                ClusterServiceOuterClass.ListClustersRequest.newBuilder()
                        .setFolderId(request.getFolderId())
                        .setPageSize(request.getPageSize());

        if (request.getFilterString() != null && !request.getFilterString().isEmpty()) {
            requestBuilder.setFilter(request.getFilterString());
        }

        if (request.getNextPageToken() != null && !request.getNextPageToken().isEmpty()) {
            requestBuilder.setPageToken(request.getNextPageToken());
        }

        ClusterServiceOuterClass.ListClustersRequest grpcRequest = requestBuilder.build();

        GrpcCallInfo callInfo = helper.createCallInfo(
                ClusterServiceGrpc.getListMethod().getFullMethodName(), metadata, grpcRequest.toString());

        ClusterServiceOuterClass.ListClustersResponse response;
        try {
            helper.logBeginCall(callInfo);
            response = stub.withDeadline(
                            Deadline.after(helper.getConnectionInfo().getRequestTimeoutInMs(), TimeUnit.MILLISECONDS))
                    .list(grpcRequest);
            helper.logSuccessEndCall(callInfo);
        } catch (StatusRuntimeException ex) {
            helper.logErrorEndCall(callInfo, ex);
            throw new IOException(String.format("Something goes wrong on calling method %s",
                    callInfo.getMethodInfo()), ex);
        }
        int count = response.getClustersCount();
        ClusterRawInfo[] clusters = new ClusterRawInfo[count];
        for (int i = 0; i < count; i++) {
            ClusterOuterClass.Cluster grpcCluster = response.getClusters(i);
            clusters[i] = new ClusterRawInfo(
                    grpcCluster.getId(),
                    grpcCluster.getName(),
                    grpcCluster.getHealth().name(),
                    grpcCluster.getStatus().name(),
                    grpcCluster.getEnvironment().name());
        }
        return new GetClustersListResponse(clusters, response.getNextPageToken());
    }

}
