package ru.yandex.direct.cloud.mdb.mysql.api.grpc;

import java.io.IOException;
import java.time.Instant;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.Timestamp;
import io.grpc.Deadline;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import yandex.cloud.api.mdb.mysql.v1.ClusterServiceGrpc;
import yandex.cloud.api.mdb.mysql.v1.ClusterServiceOuterClass;

import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListResponse;

import static io.grpc.stub.MetadataUtils.newAttachHeadersInterceptor;

/**
 * Реализация метода получение списка исходных записей выбранного лого указанного кластера облака по протоколу gRPC
 */
class GrpcLogsRecordsReader {
    private static final int TIME_TO_SHIFT_IN_SECONDS = 3600;

    static GetLogRecordsListResponse getLogRecordsList(GrpcCloudApiHelper helper, GetLogRecordsListRequest request)
            throws IOException {
        Metadata metadata = helper.createMeta();

        ClusterServiceGrpc.ClusterServiceBlockingStub stub = ClusterServiceGrpc.newBlockingStub(helper.getGrpcChannel())
                .withInterceptors(newAttachHeadersInterceptor(metadata));

        Timestamp fromTime = Timestamp.newBuilder()
                .setSeconds(request.getFrom().getEpochSecond())
                .setNanos(request.getFrom().getNano())
                .build();

        ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType serviceType = getServiceType(request);

        long currentToTimeShiftInSeconds = TIME_TO_SHIFT_IN_SECONDS;

        ClusterServiceOuterClass.ListClusterLogsResponse response;

        String nextPageToken;

        while(true) {
            Timestamp toTime = Timestamp.newBuilder()
                    .setSeconds(request.getFrom().getEpochSecond() + currentToTimeShiftInSeconds)
                    .setNanos(request.getFrom().getNano())
                    .build();
            Instant now = Instant.now();
            response = getLogs(helper, stub, metadata, fromTime, toTime, serviceType, request);
            nextPageToken = response.getNextPageToken();
            if (response.getLogsCount() > 0 || toTime.getSeconds() >= now.getEpochSecond()) {
                // Если верхня граница логов в запросе была до текущей даты, но то имеет смысл проставить флаг
                // наличия дополнительных данных, даже если cloud-api вернула его пустым. Возможно, он пришел пустой
                // не потому, что данных больше нет, а потому что при указании верхней границы даты логов мы получили
                // меньше записей, чем просили, но если верхнюю границу дату подвинуть еще, то данные появятся.
                if (nextPageToken.isEmpty() && toTime.getSeconds() < now.getEpochSecond()) {
                    nextPageToken = String.valueOf(response.getLogsCount() + 1);
                }
                break;
            }
            currentToTimeShiftInSeconds *= 2;
        }
        int count = response.getLogsCount();
        String[] rawRecords = new String[count];
        for (int i = 0; i < count; i++) {
            rawRecords[i] = response.getLogs(i).getMessageMap().get("raw");
        }
        return new GetLogRecordsListResponse(rawRecords, nextPageToken);
    }

    private static ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType getServiceType(
            GetLogRecordsListRequest request) {
        ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType serviceType;
        switch (request.getLogType()) {
            case SLOW_QUERY_LOG:
                serviceType = ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType.MYSQL_SLOW_QUERY;
                break;
            case ERROR_LOG:
                serviceType = ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType.MYSQL_ERROR;
                break;
            case GENERAL_LOG:
                serviceType = ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType.MYSQL_GENERAL;
                break;
            case AUDIT_LOG:
                serviceType = ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType.MYSQL_AUDIT;
                break;
            default:
                throw new IllegalArgumentException(String.format("Unknown request type: %s", request.getLogType()));
        }
        return serviceType;
    }

    private static ClusterServiceOuterClass.ListClusterLogsResponse getLogs(
            GrpcCloudApiHelper helper,
            ClusterServiceGrpc.ClusterServiceBlockingStub stub,
            Metadata metadata,
            Timestamp fromTime,
            Timestamp toTime,
            ClusterServiceOuterClass.ListClusterLogsRequest.ServiceType serviceType,
            GetLogRecordsListRequest request) throws IOException {

        ClusterServiceOuterClass.ListClusterLogsRequest.Builder requestBuilder =
                ClusterServiceOuterClass.ListClusterLogsRequest.newBuilder()
                        .setClusterId(request.getClusterId())
                        .setServiceType(serviceType)
                        .setPageSize(request.getPageSize())
                        .setFromTime(fromTime);

        if (toTime.getSeconds() < Instant.now().getEpochSecond()) {
            requestBuilder.setToTime(toTime);
        }

        if (request.getNextPageToken() != null && !request.getNextPageToken().isEmpty()) {
            requestBuilder.setPageToken(request.getNextPageToken());
        }

        ClusterServiceOuterClass.ListClusterLogsRequest grpcRequest = requestBuilder.build();

        GrpcCallInfo callInfo = helper.createCallInfo(
                ClusterServiceGrpc.getListLogsMethod().getFullMethodName(), metadata, grpcRequest.toString());

        ClusterServiceOuterClass.ListClusterLogsResponse response;
        try {
            helper.logBeginCall(callInfo);
            response = stub.withDeadline(
                            Deadline.after(helper.getConnectionInfo().getRequestTimeoutInMs(), TimeUnit.MILLISECONDS))
                    .listLogs(grpcRequest);
            helper.logSuccessEndCall(callInfo);
            return response;
        } catch (StatusRuntimeException ex) {
            helper.logErrorEndCall(callInfo, ex);
            throw new IOException(String.format("Something goes wrong on processing request:%n%s", request), ex);
        }
    }

}
