package ru.yandex.direct.libs.collections;

import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.AntPathMatcher;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.libs.collections.model.collection.Collection;
import ru.yandex.direct.libs.collections.model.collection.CollectionsResponse;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpData;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpDataResult;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static com.google.common.base.Preconditions.checkState;
import static java.net.URLDecoder.decode;
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Клиент к api Яндекс.Коллекций
 */
@ParametersAreNonnullByDefault
public class CollectionsClient {
    private static final Logger logger = LoggerFactory.getLogger(CollectionsClient.class);

    private static final String OWNER_LOGIN_VAR_NAME = "ownerLogin";
    private static final String COLLECTION_NAME_VAR_NAME = "collectionName";

    private static final String COLLECTION_URL_MATCHER =
            "https://**yandex.**/collections/user/{" + OWNER_LOGIN_VAR_NAME + "}/{" + COLLECTION_NAME_VAR_NAME + "}/";

    private final AntPathMatcher pathMatcher;
    private final CollectionsApi collectionsApi;

    public CollectionsClient(String baseUrl, ParallelFetcherFactory fetcherFactory, TvmIntegration tvmIntegration,
                             TvmService tvmService) {
        pathMatcher = new AntPathMatcher();

        collectionsApi = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(new CollectionsResponseConverter())
                        .build())
                // TODO(dimitrovsd): убрать TVM для вызовов метода getCollection
                .useTvm(tvmIntegration, tvmService)
                .withProfileName("collections_client")
                .withBaseUrl(baseUrl)
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"))
                .build()
                .create(CollectionsApi.class);
    }

    /**
     * Возвращает id коллекции по ее url.
     * Url коллекции должен иметь формат https://yandex.ru/collections/user/%owner_login%/%collection_name%/.
     *
     * @param collectionUrl url коллекции в нужном формате
     * @return id коллекции
     */
    @Nullable
    public String getCollectionId(String collectionUrl) {
        Map<String, String> pathVariables;
        try {
            // Разэскейпим урл, так как пользователь мог ввести уже заэскейпленный урл, а Smart клиент заэкспейпит его
            // еще раз
            pathVariables = pathMatcher.extractUriTemplateVariables(COLLECTION_URL_MATCHER, decode(collectionUrl,
                    UTF_8));
        } catch (IllegalStateException e) {
            logger.error("CollectionUrl: {} doesn't match pattern", collectionUrl);
            return null;
        }

        String ownerLogin = pathVariables.get(OWNER_LOGIN_VAR_NAME);
        String collectionName = pathVariables.get(COLLECTION_NAME_VAR_NAME);
        logger.info("CollectionUrl: {}", collectionUrl);
        return getCollectionId(ownerLogin, collectionName);
    }

    /**
     * Возвращает id коллекции по ее названию и логину владельца.
     *
     * @param ownerLogin     логин владельца коллекции
     * @param collectionName название коллекции
     * @return id коллекции
     */
    @Nullable
    private String getCollectionId(String ownerLogin, String collectionName) {
        Result<CollectionsResponse> queryResult = collectionsApi.getCollection(ownerLogin, collectionName).execute();

        CollectionsResponse response = queryResult.getSuccess();
        if (response == null) {
            return null;
        }

        List<Collection> collections = response.getCollections();
        if (collections == null || collections.isEmpty()) {
            return null;
        }
        checkState(collections.size() == 1);

        return collections.get(0).getId();
    }

    /**
     * Возвращает данные серпа для коллекции с заданным id.
     *
     * @param collectionId id коллекции
     * @return данные серпа {@link CollectionSerpData}
     */
    @Nullable
    public CollectionSerpData getCollectionSerpData(String collectionId) {
        Result<CollectionSerpData> queryResult = collectionsApi.getCollectionSerpData(collectionId).execute();

        return queryResult.getSuccess();
    }

    /**
     * Возвращает данные серпа для коллекции с заданным id, а также данные об ошибках запроса, если таковые были
     *
     * @param collectionId id коллекции
     * @return ответ в виде {@link CollectionSerpDataResult}
     */
    @Nonnull
    public CollectionSerpDataResult getCollectionSerpDataResult(String collectionId) {
        return CollectionSerpDataResult.ofResponse(collectionsApi.getCollectionSerpData(collectionId).execute());
    }
}
