package ru.yandex.direct.communication;

import java.util.List;

import com.google.protobuf.InvalidProtocolBufferException;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.types.ULong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.communication.container.CommunicationChannelItem;
import ru.yandex.direct.communication.container.CommunicationMessageData;
import ru.yandex.direct.communication.repository.record.CommunicationsChannelExtRecord;
import ru.yandex.direct.grid.schema.yt.tables.Communicationschannel;
import ru.yandex.direct.ytcomponents.service.CommunicationChannelDynContextProvider;

import static ru.yandex.direct.grid.schema.yt.Tables.COMMUNICATIONSCHANNEL;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
public class CommunicationChannelRepository {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationChannelRepository.class);

    private static final Communicationschannel COMMUNICATIONS_CHANNEL = COMMUNICATIONSCHANNEL.as("CM");
    public static final Field<ULong> CHANNEL = aliased(COMMUNICATIONS_CHANNEL.CHANNEL);
    public static final Field<ULong> MESSAGE_ID = aliased(COMMUNICATIONS_CHANNEL.MESSAGE_ID);
    public static final Field<ULong> EVENT_ID = aliased(COMMUNICATIONS_CHANNEL.EVENT_ID);
    public static final Field<ULong> UID = aliased(COMMUNICATIONS_CHANNEL.UID);
    public static final Field<String> SOURCE = aliased(COMMUNICATIONS_CHANNEL.SOURCE);
    public static final Field<ULong> CREATED = aliased(COMMUNICATIONS_CHANNEL.CREATED);
    public static final Field<ULong> EXPIRED = aliased(COMMUNICATIONS_CHANNEL.EXPIRED);
    public static final Field<String> DATA = aliased(COMMUNICATIONS_CHANNEL.DATA);
    public static final Field<ULong> TARGET_ENTITY_ID = aliased(COMMUNICATIONS_CHANNEL.TARGET_ENTITY_ID);
    public static final Field<ULong> NEXT_SHOW_TIME = aliased(COMMUNICATIONS_CHANNEL.NEXT_SHOW_TIME);

    private final CommunicationChannelDynContextProvider communicationChannelDynContextProvider;

    public CommunicationChannelRepository(
            CommunicationChannelDynContextProvider communicationChannelDynContextProvider) {
        this.communicationChannelDynContextProvider = communicationChannelDynContextProvider;
    }

    public List<CommunicationChannelItem> getMessagesByUid(Long uid, List<Long> eventIds) {
        if (eventIds.isEmpty()) {
            return List.of();
        }
        return getMessagesByIds(mapList(eventIds, eventId -> eventId + (uid << 32)));
    }

    public List<CommunicationChannelItem> getMessagesByIds(List<Long> ids) {
        var keys = mapList(ids, id -> {
            var record = new CommunicationsChannelExtRecord();
            record.setChannel(ULong.valueOf(1));
            record.setMessageid(ULong.valueOf(id));
            return record;
        });
        var records = communicationChannelDynContextProvider.getContext().lookupRows(
                keys,
                () -> new CommunicationsChannelExtRecord()
        );
        return mapList(records, CommunicationChannelItem::new);
    }

    public List<CommunicationMessageData> getCommunicationMessageByIds(List<Long> ids) {
        var keys = mapList(ids, id -> {
            var record = new CommunicationsChannelExtRecord();
            record.setChannel(ULong.valueOf(1));
            record.setMessageid(ULong.valueOf(id));
            return record;
        });
        var records = communicationChannelDynContextProvider.getContext().lookupRows(
                keys,
                () -> new CommunicationsChannelExtRecord()
        );
        return StreamEx.of(records)
                .map(r -> {
                    try {
                        return new CommunicationMessageData(r);
                    } catch (InvalidProtocolBufferException ex) {
                        logger.error("Exception in parsing message", ex);
                        return null;
                    }
                }).nonNull()
                .toList();
    }

}
