package ru.yandex.direct.communication;

import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.google.protobuf.MessageLite;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.InterruptedRuntimeException;

/**
 * Отправка событий в коммуникационный контур
 */

public class CommunicationClient {
    private final TvmIntegration tvmIntegration;
    private final DirectConfig directConfig;

    private static final ConcurrentLinkedQueue<CommunicationSession> pool = new ConcurrentLinkedQueue<>();

    private static synchronized CommunicationSession getSession() {
        var sourceData = pool.poll();
        if (sourceData != null) {
            return sourceData;
        }
        return new CommunicationSession();
    }

    private static synchronized void returnSession(CommunicationSession data) {
        pool.add(data);
    }

    public CommunicationClient(TvmIntegration tvmIntegration, DirectConfig directConfig) {
        this.tvmIntegration = tvmIntegration;
        this.directConfig = directConfig;
    }

    private CommunicationSession createSession()
            throws ExecutionException, TimeoutException {
        try {
            var session = getSession();
            session.init(directConfig, tvmIntegration);
            return session;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    public int send(List<MessageLite> messages)
            throws ExecutionException, TimeoutException {
        int result = 0;

        if (messages.isEmpty()) {
            return result;
        }

        CommunicationSession session = null;
        try {
            session = createSession();
            result = session.send(messages);
            returnSession(session);
        } catch (Exception e) {
            if (session != null) {
                session.close();
            }
        }
        return result;
    }

    public boolean send(MessageLite message)
            throws ExecutionException, TimeoutException {
        return send(List.of(message)) > 0;
    }

}
