package ru.yandex.direct.communication;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.google.protobuf.util.JsonFormat;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.ESourceType;
import ru.yandex.ads.bsyeti.libs.communications.ETargetObjectType;
import ru.yandex.ads.bsyeti.libs.communications.TDirectWebUIData;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.communications.TKeyValues;
import ru.yandex.ads.bsyeti.libs.communications.proto.TMessageData;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.Body;
import ru.yandex.direct.communication.container.web.Message;
import ru.yandex.direct.communication.container.web.Slide;
import ru.yandex.direct.communication.container.yt.CommunicationEventTableRow;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.core.entity.communication.model.ChangeAction;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.RestrinctionInfo;
import ru.yandex.direct.core.entity.communication.model.TargetEntityType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.ads.bsyeti.libs.communications.ECommunicationType.INFORMATION;
import static ru.yandex.ads.bsyeti.libs.communications.EEventType.CONDITIONAL_UPDATE;
import static ru.yandex.ads.bsyeti.libs.communications.ESourceType.RECCOMENDATION_RUNTIME_SERVICE;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class CommunicationHelper {

    private final static Logger logger = LoggerFactory.getLogger(CommunicationHelper.class);
    private final static long PAUSE_MILLIS = 10000;

    public static final ETargetObjectType entityTypeToObjectType(TargetEntityType entityType) {
        if (entityType == null) {
            return ETargetObjectType.TARGET_OBJECT_CLIENT;
        }
        switch (nvl(entityType, TargetEntityType.CLIENT)) {
            case USER:
                return ETargetObjectType.TARGET_OBJECT_USER;
            case CAMPAIGN:
                return ETargetObjectType.TARGET_OBJECT_CAMPAIGN;
            case GROUP:
                return ETargetObjectType.TARGET_OBJECT_ADGROUP;
            case BANNER:
                return ETargetObjectType.TARGET_OBJECT_AD;
            case CLIENT:
            default:
                return ETargetObjectType.TARGET_OBJECT_CLIENT;

        }
    }

    public static final RestrinctionInfo DEFAULT_RESTRICTION = new RestrinctionInfo()
            .withStopStatuses(List.of(
                    EMessageStatus.APPLY.name(),
                    EMessageStatus.REJECT.name()));
    public static final ChangeAction DEFAULT_ON_NOT_ACTUAL = new ChangeAction()
            .withAddStatuses(List.of(
                    EMessageStatus.MUTE.name()
            ));
    public static final ChangeAction DEFAULT_ON_ACTUAL = new ChangeAction()
            .withRemoveStatuses(List.of(
                    EMessageStatus.MUTE.name()
            ));
    public static final ChangeAction DEFAULT_ON_MINOR_VERSION_CHANGE = new ChangeAction()
            .withRemoveStatuses(List.of(
                    EMessageStatus.MUTE.name(),
                    EMessageStatus.APPLY.name(),
                    EMessageStatus.REJECT.name()
            ));
    public static final ChangeAction DEFAULT_ON_MAJOR_VERSION_CHANGE = new ChangeAction()
            .withUpdateNextShowTime(0L)
            .withRemoveStatuses(List.of(
                    EMessageStatus.MUTE.name(),
                    EMessageStatus.APPLY.name(),
                    EMessageStatus.REJECT.name()
            ));

    public static Optional<TMessageData> parseMessageData(byte[] data) {
        try {
            return Optional.of(TMessageData.parseFrom(data));
        } catch (InvalidProtocolBufferException e) {
            logger.error("Exception in parsing message", e);
            return Optional.empty();
        }
    }

    public static Message parseWebMessageData(TDirectWebUIData data, Long messageId) {
        if (data.getMessageFormat() == TDirectWebUIData.EMessageFormat.SIMPLE) {
            return getMessageWithFormatSimple(data, messageId);
        } else if (data.getMessageFormat() == TDirectWebUIData.EMessageFormat.RICH) {
            return getMessageWithFormatRich(data, messageId);
        } else {
            return null;
        }
    }

    private static Message getMessageWithFormatRich(TDirectWebUIData data, Long id) {
        if (!data.hasMessageText()) {
            logger.warn("Empty MessageText for message with id = " + id);
            return null;
        }
        try {
            Message message = JsonUtils.getObjectMapper().readValue(data.getMessageText(), Message.class);
            message.setId(id.toString());
            return message;
        } catch (IOException e) {
            logger.error("Exception on mapping to Message", e);
            return null;
        }
    }

    private static Message getMessageWithFormatSimple(TDirectWebUIData data, Long id) {
        List<Slide> slideList = new ArrayList<>();
        Slide slide = new Slide();
        Message result = new Message();
        Body body = new Body();

        body.setText(data.getMessageText());
        slide.setBody(body);
        slideList.add(slide);

        result.setId(id.toString());
        result.setSlides(slideList);
        return result;
    }

    public static int sendEventsFromTable(
            CommunicationClient communicationClient,
            YtOperator ytOperator, YtTable ytTable,
            long startRow, long endRow, int tryCount) {
        if (tryCount < 2) {
            return sendEventsFromTable(communicationClient, ytOperator, ytTable, startRow, endRow);
        }
        RuntimeException lastException = null;
        for (int tryIndex = 1; tryIndex <= tryCount; tryIndex++) {
            try {
                return sendEventsFromTable(communicationClient, ytOperator, ytTable, startRow, endRow);
            } catch (InterruptedRuntimeException ex) {
                logger.error(String.format("%d attempt is interrupted. Row %d. Stop sending.", tryIndex, startRow), ex);
                Thread.currentThread().interrupt();
                throw ex;
            } catch (RuntimeException ex) {
                logger.warn(String.format("%d attempt catch exception. Row %d.", tryIndex, startRow), ex);
                lastException = ex;
            }
            try {
                Thread.sleep(PAUSE_MILLIS);
            } catch (InterruptedException ex) {
                logger.error(String.format("%d attempt is interrupted. Row %d. Stop sending.", tryIndex, startRow), ex);
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(ex);
            }
        }
        logger.error(String.format("All attempts fail. Row %d. Stop sending.", startRow));
        throw lastException;
    }

    public static int sendEventsFromTable(
            CommunicationClient communicationClient,
            YtOperator ytOperator, YtTable ytTable,
            long startRow, long endRow) {
        List<String> stringEvents = new ArrayList<>((int) (endRow - startRow));

        CommunicationEventTableRow tableRow = new CommunicationEventTableRow();
        ytOperator.readTableByRowRange(ytTable, row -> stringEvents.add(row.getEvent()),
                tableRow,
                startRow,
                endRow);

        List<MessageLite> events = new ArrayList<>(stringEvents.size());
        try {
            for (String json : stringEvents) {
                events.add(parseEvent(json));
            }
        } catch (InvalidProtocolBufferException ex) {
            logger.error("Wrong event format", ex);
            throw new RuntimeException(ex);
        }
        try{
            return communicationClient.send(events);
        } catch (ExecutionException ex) {
            logger.error("Caught exception", ex);
            throw new RuntimeException(ex);
        } catch (TimeoutException ex) {
            logger.error("Timeout", ex);
            throw new RuntimeException(ex);
        }
    }

    private static MessageLite parseEvent(String json) throws InvalidProtocolBufferException {
        var builder = TCommunicationEvent.newBuilder();
        JsonFormat.parser().merge(json, builder);
        var sourceBuilder = builder.getDataBuilder().getSourceBuilder();
        sourceBuilder.setType(ESourceType.DIRECT_ADMIN);
        sourceBuilder.setId(Trace.current().getTraceId());
        return builder.build();
    }

    /**
     * Парсим параметрс с пользователями по формату type-column-cluster:path или cluster:path
     *
     * @return массив со значениями type, column, cluster, path
     */
    public static String[] parseUsers(String userTable) {
        String[] typeColumnClusterPath = userTable.split(":", 2);
        String path = typeColumnClusterPath[1];
        String type;
        String column;
        String cluster;
        String[] typeColumnCluster = typeColumnClusterPath[0].split("-", 3);
        if (typeColumnCluster.length < 3) {
            type = "uid";
            column = "Uid";
            cluster = typeColumnCluster[0];
        } else {
            type = typeColumnCluster[0];
            column = typeColumnCluster[1];
            cluster = typeColumnCluster[2];
        }
        return new String[]{type, column, cluster, path};
    }

    private static long cutNumber(long source, int bitNum) {
        var mask = (1L << bitNum) - 1;
        long result = 0;
        while (source > 0) {
            result ^= source & mask;
            source >>= bitNum;
        }
        return result;
    }

    public static long calculateMessageId(Long uid, ClientId clientId, long targetObjectId, long eventId) {
        return uid != null && uid == targetObjectId ?
                calculateMessageId(uid, targetObjectId, eventId) :
                calculateMessageId(clientId.asLong(), targetObjectId, eventId);
    }

    private static long calculateMessageId(long uid, long targetObjectId, long eventId) {
        long result = cutNumber(uid, 28);
        result <<= 20;
        result += cutNumber(targetObjectId, 20);
        result <<= 16;
        result += cutNumber(eventId, 16);
        return result;
    }

    public static TKeyValues convertJsonStringToProto(String json) {
        return Optional.ofNullable(json)
                .map(JsonUtils::fromJson)
                .map(node -> convertToKeyValueBuilder(null, node))
                .orElse(TKeyValues.newBuilder())
                .build();
    }

    private static TKeyValues.Builder convertToKeyValueBuilder(String key, JsonNode node) {
        var builder = TKeyValues.newBuilder();
        if (key != null) {
            builder.setKey(key);
        }
        if (node.isValueNode()) {
            builder.addListValue(node.asText());
            return builder;
        } else if (node.isArray()) {
            StreamEx.of(node.elements())
                    .forEach(item -> builder.addMapValue(convertToKeyValueBuilder(
                            null, item
                    )));
        } else {
            StreamEx.of(node.fields())
                    .sortedBy(Map.Entry::getKey)
                    .forEach(entry -> builder.addMapValue(convertToKeyValueBuilder(
                            entry.getKey(), entry.getValue()
                    )));
        }
        return builder;
    }

    public static MessageLite buildApplyEvent(
            ActionTarget actionTarget,
            CommunicationEventVersion version,
            AdditionalInfoContainer additionalInfo
    ) {
        return buildEventWithStatus(EMessageStatus.APPLY, actionTarget, version, additionalInfo);
    }

    public static MessageLite buildEventWithStatus(
            EMessageStatus status,
            ActionTarget actionTarget,
            CommunicationEventVersion version,
            AdditionalInfoContainer additionalInfo
    ) {
        var clientIdOrUserId = TargetEntityType.USER.equals(version.getTargetEntityType()) ?
                additionalInfo.getUserId().orElse(null) :
                additionalInfo.getClientId().map(ClientId::asLong).orElse(null);
        var nextShowTime = additionalInfo.getCurrentTimeStamp().get() + nextShowDelay(version, status);
        var onMinorVersionChange = TCommunicationEventData.TAction.newBuilder()
                .setUpdateNextShowTime(nextShowTime);
        var onMinorChangeRemoveStatuses = nvl(nvl(version.getOnMinorVersionChange(),
                DEFAULT_ON_MINOR_VERSION_CHANGE).getRemoveStatuses(),
                Collections.emptyList());
        if (!onMinorChangeRemoveStatuses.contains(status)) {
            onMinorVersionChange.addAddStatuses(status);
        }
        return TCommunicationEvent.newBuilder()
                .addUids(clientIdOrUserId)
                .setTimestamp(additionalInfo.getCurrentTimeStamp().orElse(null))
                .setData(TCommunicationEventData.newBuilder()
                        .setType(CONDITIONAL_UPDATE)
                        .setCommunicationType(INFORMATION)
                        .setId(actionTarget.getEventId().intValue())
                        .setTargetEntityId(actionTarget.getTargetObjectId())
                        .setSource(TEventSource.newBuilder()
                                .setType(RECCOMENDATION_RUNTIME_SERVICE)
                                .setId(version.getIter()))
                        .setConditionalUpdate(TCommunicationEventData.TConditionalUpdate.newBuilder().addConditionAndUpdate(
                                TCommunicationEventData.TConditionalUpdate.TConditionAndUpdate.newBuilder()
                                        .setConditionMajorVersion(actionTarget.getMajorDataVersion())
                                        .setConditionMinorVersion(actionTarget.getMinorDataVersion())
                                        .setAction(TCommunicationEventData.TAction.newBuilder()
                                                .setUpdateNextShowTime(nextShowTime)
                                                .addAddStatuses(status))).addConditionAndUpdate(
                                TCommunicationEventData.TConditionalUpdate.TConditionAndUpdate.newBuilder()
                                        .setConditionMajorVersion(actionTarget.getMajorDataVersion())
                                        .setAction(onMinorVersionChange))))
                .build();
    }

    private static long nextShowDelay(CommunicationEventVersion version, EMessageStatus status) {
        long delay = nvl(version.getTimeToLive(), 0L);
        return version.getTimeToLiveByStatus() == null ? delay :
                version.getTimeToLiveByStatus().getOrDefault(status.name(), delay);
    }

    private static final String NUMBER_FORMAT_PREFIX = "Number_";

    /**
     * Преобразует значение к строке.
     * По дефолту используется стандартный toString
     * Поддерживается преобразования числа по шаблону. Тип в зависимости от шаблона выглядит например Number_#,00
     * @param value значение
     * @param transformType способ преобразования
     * @return Строковое представление значения
     */
    public static String transform(Object value, String transformType) {
        if (value == null) {
            return null;
        } else if (transformType == null) {
            return value.toString();
        } else if (value instanceof Number && transformType.startsWith(NUMBER_FORMAT_PREFIX)) {
            DecimalFormat df = new DecimalFormat(transformType.substring(NUMBER_FORMAT_PREFIX.length()));
            return df.format(value);
        } else {
            return value.toString();
        }
    }

    private static final String SUM_NUMBER_FORMAT_PREFIX = "Sum_Number_";
    private static final String JOIN_WITH_SEPARATOR = "Join_";

    public static String merge(List<String> values, String mergeType) {
        if (values == null) {
            return null;
        } else if (mergeType == null) {
            return values.stream().findFirst().orElse(null);
        } else if (mergeType.startsWith(SUM_NUMBER_FORMAT_PREFIX)) {
            var format = mergeType.substring(SUM_NUMBER_FORMAT_PREFIX.length());
            DecimalFormat df = new DecimalFormat(format);
            return df.format(StreamEx.of(values)
                    .mapToDouble(v -> {
                        try {
                            return df.parse(v).doubleValue();
                        } catch (ParseException e) {
                            return 0;
                        }
                    })
                    .sum());
        } else if (mergeType.startsWith(JOIN_WITH_SEPARATOR)) {
            var separator = mergeType.substring(JOIN_WITH_SEPARATOR.length());
            return String.join(separator, values);
        } else {
            return values.stream().findFirst().orElse(null);
        }
    }

}
