package ru.yandex.direct.communication;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

import com.google.protobuf.MessageLite;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerWriteResponse;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;

public class CommunicationSession implements AutoCloseable {

    private static class Message {
        byte[] bytes;
        CompletableFuture<ProducerWriteResponse> future;

        Message(MessageLite message) {
            bytes = message.toByteArray();
            future = new CompletableFuture<>();
        }

        int getSize() {
            return bytes.length;
        }
    }

    private static final Long SENDING_TIMEOUT_SEC = 60L;
    private static final Long SENDING_LIMIT = (long) (8 * 1024 * 1024);
    private static final String CONFIG_SECTION = "communication_platform_lb_settings_sending";
    private static final Logger logger = LoggerFactory.getLogger(CommunicationSession.class);
    private final AtomicLong bytesInProgress = new AtomicLong(0);
    private final Queue<Message> messageQueue = new LinkedList<>();
    final String sourceUUID;
    AsyncProducer producer;

    CommunicationSession() {
        this.sourceUUID = UUID.randomUUID().toString();
    }

    private AsyncProducerConfig getProducerConfig(Supplier<Credentials> logBrokerCredentialsSupplier,
                                                  String writeTopic, String sourceId) {
        byte[] sourceIdBytes = sourceId.getBytes(StandardCharsets.UTF_8);
        AsyncProducerConfig.Builder builder = AsyncProducerConfig
                .builder(writeTopic, sourceIdBytes)
                .setCodec(CompressionCodec.RAW)
                .setCredentialsProvider(logBrokerCredentialsSupplier);
        return builder.build();
    }

    private Supplier<Credentials> getSupplier(TvmService targetService, TvmIntegration tvmIntegration) {
        return () -> Credentials.tvm(tvmIntegration.getTicket(targetService));
    }

    private <T> T futureComplete(CompletableFuture<T> completableFuture)
            throws ExecutionException, TimeoutException {
        try {
            return completableFuture.get(SENDING_TIMEOUT_SEC, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    public void init(DirectConfig directConfig, TvmIntegration tvmIntegration)
            throws InterruptedException, ExecutionException, TimeoutException {

        if (producer != null) return;

        DirectConfig producerSectionConfig = directConfig.getBranch(CONFIG_SECTION);

        String writeTopic = producerSectionConfig.getString("write_topic");
        TvmService targetService = TvmService.fromStringStrict(
                producerSectionConfig.getString("tvm_service_name"));

        var sourceId = "CommunicationClient:" + SystemUtils.hostname() + ":" + sourceUUID;
        AsyncProducerConfig producerConfig = getProducerConfig(
                getSupplier(targetService, tvmIntegration), writeTopic, sourceId);
        ProxyBalancer proxyBalancer = new ProxyBalancer(producerSectionConfig.getString("host"));

        producer = new LogbrokerClientFactory(proxyBalancer).asyncProducer(producerConfig);

        futureComplete(producer.init());
    }

    private CompletableFuture<ProducerWriteResponse> sendMessage(MessageLite message) {
        var internalMessage = new Message(message);
        addMessageToQueue(internalMessage);
        sendNextMessageFromQueue();
        return internalMessage.future;
    }

    synchronized private void addMessageToQueue(Message message) {
        messageQueue.add(message);
    }

    synchronized private void sendNextMessageFromQueue() {
        var message = messageQueue.peek();
        if (message == null) return;

        if (bytesInProgress.get() + message.getSize() < SENDING_LIMIT) {
            bytesInProgress.addAndGet(message.getSize());
            messageQueue.remove();
            producer.write(message.bytes).handleAsync((result, error) -> {
                bytesInProgress.addAndGet(-message.getSize());
                if (error != null) {
                    message.future.completeExceptionally(error);
                } else {
                    message.future.complete(result);
                    sendNextMessageFromQueue();
                }
                return result;
            });
        }
    }

    public int send(List<MessageLite> messages) throws ExecutionException, TimeoutException {

        var futureResults = StreamEx.of(messages)
                .map(this::sendMessage)
                .toList();

        List<Long> alreadyWrittenSeqNo = new ArrayList<>();
        List<Long> writtenSeqNo = new ArrayList<>();

        for (var futureResult : futureResults) {
            ProducerWriteResponse response = futureComplete(futureResult);
            logger.debug("SeqNo = {}", response.getSeqNo());
            if (response.isAlreadyWritten()) {
                alreadyWrittenSeqNo.add(response.getSeqNo());
            } else {
                writtenSeqNo.add(response.getSeqNo());
            }
        }
        if (!alreadyWrittenSeqNo.isEmpty()) {
            logger.warn("{} records were already written. Already written SeqNo: {}",
                    alreadyWrittenSeqNo.size(), alreadyWrittenSeqNo);
            if (!writtenSeqNo.isEmpty()) {
                logger.warn("Only {} records were written. SeqNo: {}", writtenSeqNo.size(), writtenSeqNo);
            }
        }
        return writtenSeqNo.size();
    }

    public void close() {
        if (producer != null) {
            producer.close();
            try {
                producer.closeFuture().get(SENDING_TIMEOUT_SEC, TimeUnit.SECONDS);
            } catch (Exception e) {
                logger.warn("Couldn't close async producer gracefully", e);
            } finally {
                producer = null;
            }
        }
    }

}
