package ru.yandex.direct.communication.config;

import org.asynchttpclient.AsyncHttpClient;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.autobudget.restart.AutobudgetRestartConfiguration;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.inventory.CommunicationInventoryClient;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.db.config.DbConfigException;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.useractionlog.TableNames;
import ru.yandex.direct.useractionlog.db.DbConfigClickHouseManager;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.db.ShardReplicaChooser;
import ru.yandex.direct.useractionlog.db.ShardReplicaChooserLazyProxy;
import ru.yandex.direct.useractionlog.reader.LastAutoUpdatedSettingsActionLogReader;

@Configuration
@Import(value = {
        EssentialConfiguration.class, AutobudgetRestartConfiguration.class
})
@ComponentScan(
        basePackages = "ru.yandex.direct.communication",
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
        }
)
public class CommunicationConfiguration {
    public static final String COMMUNICATION_INVENTORY_CLIENT = "communicationInventoryClient";

    @Bean
    public CommunicationClient communicationClient(TvmIntegration tvmIntegration, DirectConfig directConfig) {
        return new CommunicationClient(tvmIntegration, directConfig);
    }

    @Bean(name = COMMUNICATION_INVENTORY_CLIENT)
    public CommunicationInventoryClient communicationInventoryClient(
            @Value("${communication_inventory_client.url}") String baseUrl,
            AsyncHttpClient asyncHttpClient
    ) {
        ParallelFetcherFactory fetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        return new CommunicationInventoryClient(baseUrl, fetcherFactory);
    }

    @Bean
    public ShardReplicaChooser userActionLogShardReplicaChooser(
            DbConfigFactory dbConfigFactory,
            DatabaseWrapperProvider databaseWrapperProvider) {
        return new ShardReplicaChooserLazyProxy(() -> {
            try {
                return DbConfigClickHouseManager
                        .create(dbConfigFactory, SimpleDb.PPCHOUSE_PPC.toString(), databaseWrapperProvider);
            } catch (DbConfigException | DbConfigClickHouseManager.ValidationError error) {
                throw new IllegalStateException(
                        "UserActionLogReader can't be initialized because of error in dbconfig: " + error.getMessage());
            }
        });
    }

    @Bean
    public LastAutoUpdatedSettingsActionLogReader lastAutoUpdatedSettingsActionLogReader(
            ShardReplicaChooser userActionLogShardReplicaChooser) {
        return new LastAutoUpdatedSettingsActionLogReader(
                new ReadActionLogTable(userActionLogShardReplicaChooser::getForReading,
                        TableNames.READ_USER_ACTION_LOG_TABLE));
    }
}
