package ru.yandex.direct.communication.container;

import java.util.List;
import java.util.Optional;

import ru.yandex.ads.bsyeti.libs.communications.ECommunicationType;
import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.proto.TMessageData;
import ru.yandex.direct.communication.container.web.WebMessage;
import ru.yandex.direct.communication.repository.record.CommunicationsChannelExtRecord;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;

import static ru.yandex.direct.communication.CommunicationChannelRepository.CHANNEL;
import static ru.yandex.direct.communication.CommunicationChannelRepository.CREATED;
import static ru.yandex.direct.communication.CommunicationChannelRepository.DATA;
import static ru.yandex.direct.communication.CommunicationChannelRepository.EVENT_ID;
import static ru.yandex.direct.communication.CommunicationChannelRepository.EXPIRED;
import static ru.yandex.direct.communication.CommunicationChannelRepository.MESSAGE_ID;
import static ru.yandex.direct.communication.CommunicationChannelRepository.SOURCE;
import static ru.yandex.direct.communication.CommunicationChannelRepository.UID;
import static ru.yandex.direct.communication.CommunicationHelper.parseMessageData;
import static ru.yandex.direct.ytwrapper.YtTableUtils.bytesValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

public class CommunicationChannelItem {

    private final int channel;
    private final long messageId;
    private final int created;
    private final long eventId;
    private final int expired;
    private final String source;
    private final long uid;
    private final List<EMessageStatus> statuses;
    private final ECommunicationType communicationType;
    private final WebMessage webMessage;

    public CommunicationChannelItem(int channel, long messageId, int created, long eventId, int expired,
                                    String source, long uid, List<EMessageStatus> statuses,
                                    ECommunicationType communicationType, WebMessage webMessage) {
        this.channel = channel;
        this.messageId = messageId;
        this.created = created;
        this.eventId = eventId;
        this.expired = expired;
        this.source = source;
        this.uid = uid;
        this.statuses = statuses;
        this.communicationType = communicationType;
        this.webMessage = webMessage;
    }

    public CommunicationChannelItem(CommunicationsChannelExtRecord record) {
        channel = record.getChannel().intValue();
        messageId = record.getMessageid().longValue();
        created = record.getCreated().intValue();
        eventId = record.getEventid().longValue();
        expired = record.getExpired().intValue();
        source = record.getSource();
        uid = record.getUid().longValue();

        Optional<TMessageData> messageData = parseMessageData(record.getDataExt());
        statuses = messageData.map(TMessageData::getStatusList).orElse(null);
        communicationType = messageData.map(TMessageData::getCommunicationType).orElse(null);
        webMessage = messageData
                .filter(TMessageData::hasDirectWebData)
                .map(TMessageData::getDirectWebData)
                .map(d -> new WebMessage(d, messageId))
                .orElse(null);
    }

    public CommunicationChannelItem(UnversionedRow row, TableSchema schema) {
        channel = longValueGetter(schema, CHANNEL).apply(row).intValue();
        messageId = longValueGetter(schema, MESSAGE_ID).apply(row);
        created = longValueGetter(schema, CREATED).apply(row).intValue();
        eventId = longValueGetter(schema, EVENT_ID).apply(row);
        expired = longValueGetter(schema, EXPIRED).apply(row).intValue();
        source = stringValueGetter(schema, SOURCE).apply(row);
        uid = longValueGetter(schema, UID).apply(row);

        byte[] data = bytesValueGetter(schema, DATA).apply(row);
        Optional<TMessageData> messageData = parseMessageData(data);
        statuses = messageData.map(TMessageData::getStatusList).orElse(null);
        communicationType = messageData.map(TMessageData::getCommunicationType).orElse(null);
        webMessage = messageData
                .filter(TMessageData::hasDirectWebData)
                .map(TMessageData::getDirectWebData)
                .map(d -> new WebMessage(d, messageId))
                .orElse(null);
    }

    public int getChannel() {
        return channel;
    }

    public long getMessageId() {
        return messageId;
    }

    public int getCreated() {
        return created;
    }

    public long getEventId() {
        return eventId;
    }

    public int getExpired() {
        return expired;
    }

    public String getSource() {
        return source;
    }

    public long getUid() {
        return uid;
    }

    public List<EMessageStatus> getStatuses() {
        return statuses;
    }

    public ECommunicationType getCommunicationType() {
        return communicationType;
    }

    public WebMessage getWebMessage() {
        return webMessage;
    }
}
