package ru.yandex.direct.communication.container;

import java.util.Optional;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.ads.bsyeti.libs.communications.EChannel;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.communications.proto.TMessageData;
import ru.yandex.direct.communication.repository.record.CommunicationsChannelExtRecord;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;

import static ru.yandex.direct.communication.CommunicationChannelRepository.CHANNEL;
import static ru.yandex.direct.communication.CommunicationChannelRepository.DATA;
import static ru.yandex.direct.communication.CommunicationChannelRepository.EVENT_ID;
import static ru.yandex.direct.communication.CommunicationChannelRepository.MESSAGE_ID;
import static ru.yandex.direct.communication.CommunicationChannelRepository.NEXT_SHOW_TIME;
import static ru.yandex.direct.communication.CommunicationChannelRepository.SOURCE;
import static ru.yandex.direct.communication.CommunicationChannelRepository.TARGET_ENTITY_ID;
import static ru.yandex.direct.communication.CommunicationChannelRepository.UID;
import static ru.yandex.direct.ytwrapper.YtTableUtils.bytesValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;

public class CommunicationMessageData {

    private final EChannel channel;
    private final long messageId;
    private final long userId;
    private final long targetEntityId;
    private final long eventId;
    private final long nextShowTime;
    private final TEventSource source;
    private final TMessageData messageData;

    public CommunicationMessageData(CommunicationsChannelExtRecord record) throws InvalidProtocolBufferException {
        channel = EChannel.forNumber(record.getChannel().intValue());
        messageId = record.getMessageid().longValue();
        userId = record.getUid().longValue();
        targetEntityId = record.getTargetentityid().longValue();
        eventId = record.getEventid().longValue();
        nextShowTime = record.getNextshowtime().longValue();
        source = TEventSource.parseFrom(record.getSource().getBytes());
        messageData = TMessageData.parseFrom(record.getDataExt());
    }

    public CommunicationMessageData(UnversionedRow row, TableSchema schema) throws InvalidProtocolBufferException {
        channel = EChannel.forNumber(longValueGetter(schema, CHANNEL).apply(row).intValue());
        messageId = longValueGetter(schema, MESSAGE_ID).apply(row);
        userId = longValueGetter(schema, UID).apply(row);
        targetEntityId = longValueGetter(schema, TARGET_ENTITY_ID).apply(row);
        eventId = longValueGetter(schema, EVENT_ID).apply(row);
        nextShowTime = Optional.ofNullable(longValueGetter(schema, NEXT_SHOW_TIME).apply(row)).orElse(0L);
        source = TEventSource.parseFrom(
                bytesValueGetter(schema, SOURCE).apply(row));
        messageData = TMessageData.parseFrom(
                bytesValueGetter(schema, DATA).apply(row));
    }

    public CommunicationMessageData(EChannel channel, long messageId, long userId, long targetEntityId, long eventId,
                                    Long nextShowTime, TEventSource eventSource, TMessageData messageData) {
        this.channel = channel;
        this.messageId = messageId;
        this.userId = userId;
        this.targetEntityId = targetEntityId;
        this.eventId = eventId;
        this.nextShowTime = Optional.ofNullable(nextShowTime).orElse(0L);
        this.source = eventSource;
        this.messageData = messageData;
    }

    public EChannel getChannel() {
        return channel;
    }

    public long getMessageId() {
        return messageId;
    }

    public long getUserId() {
        return userId;
    }

    public long getTargetEntityId() {
        return targetEntityId;
    }

    public long getEventId() {
        return eventId;
    }

    public long getNextShowTime() {
        return nextShowTime;
    }

    public TEventSource getSource() {
        return source;
    }

    public TMessageData getMessageData() {
        return messageData;
    }
}
