package ru.yandex.direct.communication.facade;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.CommunicationMessageData;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.container.web.CommunicationMessageGroup;
import ru.yandex.direct.communication.container.web.SlotMessageId;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.communication.model.inventory.Response;
import ru.yandex.direct.communication.model.inventory.SlotResponse;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.communication.model.ButtonConfig;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.useractionlog.model.AutoUpdatedSettingsEvent;

import static java.util.function.Function.identity;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
@ParametersAreNonnullByDefault
public class CommunicationEventVersionProcessingFacade {

    public static final String DEFAULT_IMPLEMENTATION = "default";
    public static final String ON_CLIENT_CREATE_TRIGGER = "onClientCreate";
    public static final String ON_CLIENT_MCC_REQUEST_TRIGGER = "onClientMccRequest";

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventVersionProcessingFacade.class);

    private final Map<String, CommunicationEventVersionsProcessor> processors;
    private final Map<String, ActualChecker> actualCheckers;
    private final Map<String, StatusChecker> statusCheckers;
    private final Map<String, CommunicationMessageFormatter> formatters;
    private final Map<String, CommunicationMessageGroupFormatter> groupFormatters;
    private final Map<String, ActionHandler> actionHandlers;
    private final Map<String, MessageSender> senders;
    private final Map<String, ContextEntityKeyFinder<Long>> entityKeyFinders;
    private final CampaignRepository campaignRepository;
    private final Map<String, AutoUpdatedSettingsEventFormatter> eventFormatters;

    @Autowired
    public CommunicationEventVersionProcessingFacade(
            List<CommunicationEventVersionsProcessor> processors,
            List<ActualChecker> actualCheckers,
            List<StatusChecker> statusCheckers,
            List<CommunicationMessageFormatter> formatters,
            List<CommunicationMessageGroupFormatter> groupFormatters,
            List<ActionHandler> actionHandlers,
            List<MessageSender> senders,
            List<ContextEntityKeyFinder> entityKeyFinders,
            CampaignRepository campaignRepository,
            List<AutoUpdatedSettingsEventFormatter> eventFormatters
    ) {
        this.processors = listToMap(processors, CommunicationEventVersionsProcessor::getName);
        this.actualCheckers = listToMap(actualCheckers, ActualChecker::getName);
        this.statusCheckers = listToMap(statusCheckers, StatusChecker::getName);
        this.formatters = listToMap(formatters, CommunicationMessageFormatter::getName);
        this.groupFormatters = listToMap(groupFormatters, CommunicationMessageGroupFormatter::getName);
        this.actionHandlers = listToMap(actionHandlers, ActionHandler::getName);
        this.senders = listToMap(senders, MessageSender::getName);
        this.entityKeyFinders = listToMap(entityKeyFinders, ContextEntityKeyFinder::getName);
        this.campaignRepository = campaignRepository;
        this.eventFormatters = listToMap(eventFormatters, AutoUpdatedSettingsEventFormatter::getName);
    }

    public void processOnTrigger(
            String triggerName,
            List<CommunicationEventVersion> eventVersions,
            AdditionalInfoContainer additionalInfo
    ) {
        try {
            var processor = getImplementationByName(triggerName, processors);
            processor.process(
                    additionalInfo,
                    eventVersions,
                    triggerName);
        } catch (RuntimeException e) {
            logger.error("Communication event version processing failed. Processor {}.",
                    triggerName, e);
        }
    }

    public void processOnClientCreateEvents(
            AdditionalInfoContainer additionalInfo,
            String onClientCreateAction,
            List<CommunicationEventVersion> eventVersions) {
        try {
            var processor = getImplementationByName(onClientCreateAction, processors);
            processor.process(additionalInfo, eventVersions, ON_CLIENT_CREATE_TRIGGER);
        } catch (RuntimeException e) {
            logger.error("Communication event version processing failed. Processor {}, clientId {} .",
                    onClientCreateAction, additionalInfo.getClientId(), e);
        }
    }

    private <T> T getImplementationByName(String name, Map<String, T> implMap) {
        Assert.isTrue(!isBlank(name), "empty implementation name");

        T implementation = implMap.get(name);
        Assert.notNull(implementation, String.format("unknown implementation name %s", name));

        return implementation;
    }

    public List<CommunicationMessage> handle(
            Response response,
            Map<Long, Map<Long, CommunicationEventVersion>> eventVersions,
            Map<Long, CommunicationMessageData> caesarDataByMessageId,
            Map<Long, Slot> slotById,
            AdditionalInfoContainer additionalInfo) {
        var recommendByMessageId = StreamEx.of(response.getSlotResponsesList())
                .flatCollection(SlotResponse::getObjectEventDataList)
                .filter(data -> eventVersions.containsKey(data.getEventId()) &&
                        eventVersions.get(data.getEventId()).containsKey(data.getEventVersionId()))
                .toMap(
                        SlotMessageId::of,
                        identity()
                );
        var versionByMessageId = EntryStream.of(recommendByMessageId)
                .mapValues(r -> eventVersions.get(r.getEventId()).get(r.getEventVersionId()))
                .toMap();
        var isActualByMessageId = EntryStream.of(versionByMessageId)
                .invert()
                .mapKeys(v -> Optional.ofNullable(v.getCheckActual()).orElse(DEFAULT_IMPLEMENTATION))
                .mapValues(recommendByMessageId::get)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapToValue((actualCheckName, recs) -> getImplementationByName(actualCheckName, actualCheckers)
                        .check(recs, versionByMessageId, additionalInfo))
                .values()
                .flatMapToEntry(identity())
                .toMap();
        var notHiddenMessageId = EntryStream.of(versionByMessageId)
                .invert()
                .mapKeys(v -> Optional.ofNullable(v.getCheckStatus()).orElse(DEFAULT_IMPLEMENTATION))
                .mapValues(recommendByMessageId::get)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapToValue((statusCheckName, recs) -> getImplementationByName(statusCheckName, statusCheckers)
                        .filterHidden(recs, caesarDataByMessageId, versionByMessageId, slotById, isActualByMessageId, additionalInfo))
                .flatCollection(Map.Entry::getValue)
                .distinct()
                .toSet();

        additionalInfo.withContextCampaignsSupplier(info -> listToMap(
                campaignRepository.getCampaigns(
                        info.getShard().get(),
                        EntryStream.of(versionByMessageId)
                                .filterKeys(notHiddenMessageId::contains)
                                .mapValues(CommunicationEventVersion::getCidFinders)
                                .nonNullValues()
                                .flatMapValues(Collection::stream)
                                .invert()
                                .mapValues(recommendByMessageId::get)
                                .sortedBy(Map.Entry::getKey)
                                .collapseKeys()
                                .mapToValue((finderName, recs) -> getImplementationByName(finderName, entityKeyFinders)
                                        .find(recs, versionByMessageId, slotById, additionalInfo))
                                .values()
                                .flatCollection(identity())
                                .distinct()
                                .toSet()),
                Campaign::getId));

        var resultByMessageId = EntryStream.of(versionByMessageId)
                .filterKeys(notHiddenMessageId::contains)
                .invert()
                .mapKeys(v -> Optional.ofNullable(v.getFormatName()).orElse(DEFAULT_IMPLEMENTATION))
                .mapValues(recommendByMessageId::get)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapToValue((formatName, recs) -> getImplementationByName(formatName, formatters)
                        .format(recs, caesarDataByMessageId, versionByMessageId, slotById, isActualByMessageId, additionalInfo))
                .values()
                .flatMapToEntry(identity())
                .toMap();
        return StreamEx.of(response.getSlotResponsesList())
                .flatCollection(SlotResponse::getObjectEventDataList)
                .map(SlotMessageId::of)
                .filter(notHiddenMessageId::contains)
                .map(resultByMessageId::get)
                .nonNull()
                .toList();
    }

    public List<CommunicationMessageGroup> groupCommunicationMessages(
            List<CommunicationMessage> messages,
            Map<SlotMessageId, CommunicationEventVersion> versionBySlotMessageId,
            AdditionalInfoContainer additionalInfo
    ) {
        var messageById = listToMap(messages, SlotMessageId::of);
        return EntryStream.of(versionBySlotMessageId)
                .invert()
                .mapToKey((version, slotMessageId) -> Pair.of(slotMessageId.getSlotId(),
                        Optional.ofNullable(version.getGroupFormatName()).orElse(DEFAULT_IMPLEMENTATION)))
                .mapValues(messageById::get)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapToValue((formatNameWithSlot, mes) -> getImplementationByName(
                        formatNameWithSlot.getRight(), groupFormatters)
                        .format(mes, versionBySlotMessageId, additionalInfo))
                .flatCollection(Map.Entry::getValue)
                .toList();
    }

    public ActionResult confirmAction(
            AdditionalInfoContainer additionalInfo,
            Map<Long, ActionTarget> targetByMessageId,
            Map<Long, CommunicationMessage> messageById,
            Map<Long, CommunicationEventVersion> versionByMessageId,
            Slot slot,
            long buttonId) {
        targetByMessageId.forEach((id, target) -> {
            if (target.getInventoryAdditionalData() != null) {
                return;
            }
            target.setInventoryAdditionalData(Optional.ofNullable(messageById.get(id))
                    .map(CommunicationMessage::getInventoryAdditionalData)
                    .orElse(null));
        });
        return EntryStream.of(versionByMessageId)
                .invert()
                .mapKeys(v -> Optional.ofNullable(v.getButtonConfigs())
                        .map(t -> t.get((int) buttonId))
                        .map(ButtonConfig::getHandlerName).orElse(DEFAULT_IMPLEMENTATION))
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapKeyValue((handlerName, messageIds) -> getImplementationByName(handlerName, actionHandlers)
                        .confirm(listToMap(messageIds, identity(), targetByMessageId::get),
                                additionalInfo, slot, buttonId, messageById, versionByMessageId))
                .nonNull()
                .collect(ActionResult::createEmpty, ActionResult::merge, ActionResult::merge);
    }

    public void sendToChannel(
            String channelName,
            List<CommunicationMessage> messages,
            AdditionalInfoContainer additionalInfo,
            Map<Long, CommunicationEventVersion> versionByMessageId
    ) {
        getImplementationByName(channelName, senders).send(messages, additionalInfo, versionByMessageId);
    }

    public List<CommunicationMessage> formatAutoUpdatedSettingsEvent(AutoUpdatedSettingsEvent event,
                                                               CommunicationEventVersion version,
                                                               List<Slot> slots,
                                                               AdditionalInfoContainer additionalInfo){
        var formatName = nvl(version.getFormatName(), DEFAULT_IMPLEMENTATION);
        var formatter = getImplementationByName(formatName, eventFormatters);
        slots = filterList(slots, slot -> version.getSlots().contains(slot.getId()));
        return formatter.format(event, version, slots, additionalInfo);
    }
}
