package ru.yandex.direct.communication.facade.impl.actions;

import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.facade.impl.logging.RecommendationLogger;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.result.Result;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.communication.CommunicationHelper.buildApplyEvent;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

abstract class AbstractApplyBudgetRecommendation extends AbstractClientAction<ActionResultImpl> {

    private static final Logger logger = LoggerFactory.getLogger(AbstractApplyBudgetRecommendation.class);

    protected final ShardHelper shardHelper;
    protected final CampaignRepository campaignRepository;

    protected AbstractApplyBudgetRecommendation(
            CommunicationClient communicationClient,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository
    ) {
        super(communicationClient);
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
    }

    @Override
    protected ActionResultImpl failByReadOnly(Slot slot, Collection<ActionTarget> targets) {
        return new ActionResultImpl(emptyList(), mapList(targets, ActionTarget::getTargetObjectId));
    }

    @Override
    protected ActionResultImpl confirmInternal(
            List eventsToSend,
            ActionTarget target,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            long buttonId,
            CommunicationMessage message,
            CommunicationEventVersion eventVersion
    ) {
        String failReason = getFailReason(target, message, eventVersion);
        if (failReason != null) {
            RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), failReason);
            return new ActionResultImpl(emptyList(), List.of(target.getTargetObjectId()));
        }

        return applyUpdatedBudget(eventsToSend, target, additionalInfo, slot, message, eventVersion);
    }

    abstract ActionResultImpl applyUpdatedBudget(List eventsToSend, ActionTarget target,
                                                 AdditionalInfoContainer additionalInfo,
                                                 Slot slot, CommunicationMessage message,
                                                 CommunicationEventVersion eventVersion);

    protected ActionResultImpl getResult(
            Result result, Long campaignId, List eventsToSend, ActionTarget target,
            AdditionalInfoContainer additionalInfo, Slot slot, CommunicationEventVersion eventVersion) {
        if (!result.isSuccessful()) {
            logger.info("Failed with errors " + result.getErrors());
        }
        return getResult(result.isSuccessful(), campaignId, eventsToSend, target, additionalInfo, slot, eventVersion);
    }

    protected ActionResultImpl getResult(
            boolean isSuccessful, Long campaignId, List eventsToSend, ActionTarget target,
            AdditionalInfoContainer additionalInfo, Slot slot, CommunicationEventVersion eventVersion) {
        if (isSuccessful) {
            RecommendationLogger.logApply(additionalInfo, target, slot.getId());
            eventsToSend.add(buildApplyEvent(target, eventVersion, additionalInfo));
            return new ActionResultImpl(List.of(campaignId), emptyList());
        }

        RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), "internal");
        return new ActionResultImpl(emptyList(), List.of(campaignId));
    }

}
