package ru.yandex.direct.communication.facade.impl.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.google.protobuf.MessageLite;
import one.util.streamex.EntryStream;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionHandler;
import ru.yandex.direct.communication.facade.ActionResult;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;

abstract public class AbstractClientAction<Result extends ActionResult<Result>> implements ActionHandler<Result> {

    protected final CommunicationClient communicationClient;

    public AbstractClientAction(CommunicationClient communicationClient) {
        this.communicationClient = communicationClient;
    }

    @Override
    public Result confirm(
            Map<Long, ActionTarget> targets,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            long buttonId,
            Map<Long, CommunicationMessage> messageById,
            Map<Long, CommunicationEventVersion> eventVersionByMessageId
    ) {
        if (!additionalInfo.getCanUserWrite().orElse(false)) {
            return failByReadOnly(slot, targets.values());
        }
        return confirmInternal(targets, additionalInfo, slot, buttonId, messageById, eventVersionByMessageId);
    }

    abstract protected Result failByReadOnly(
            Slot slot,
            Collection<ActionTarget> targets
    );

    protected Result confirmInternal(
            Map<Long, ActionTarget> targets,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            long buttonId,
            Map<Long, CommunicationMessage> messageById,
            Map<Long, CommunicationEventVersion> eventVersionByMessageId
    ) {
        List<MessageLite> eventsToSend = new ArrayList<>();
        var resultIterator =  EntryStream.of(targets)
                .map(entry -> {
                    var mId = entry.getKey();
                    return confirmInternal(eventsToSend,
                            entry.getValue(),
                            additionalInfo, slot, buttonId,
                            messageById.get(mId),
                            eventVersionByMessageId.get(mId)
                    );
                })
                .iterator();
        var result = resultIterator.next();
        while (resultIterator.hasNext()) {
            result = result.merge(resultIterator.next());
        }

        try{
            communicationClient.send(eventsToSend);
        } catch (ExecutionException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
        return result;
    }

    abstract protected Result confirmInternal(
            List eventsToSend,
            ActionTarget target,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            long buttonId,
            CommunicationMessage message,
            CommunicationEventVersion eventVersion
    );

    protected String getFailReason(
            ActionTarget target,
            CommunicationMessage message,
            CommunicationEventVersion eventVersion
    ) {
        if (eventVersion == null || !CommunicationEventVersionStatus.ACTIVE.equals(eventVersion.getStatus())) {
            return "not_active_event";
        } else if (message == null) {
            return "not_actual";
        } else if (message.getMajorVersion() != target.getMajorDataVersion() ||
                message.getMinorVersion() != target.getMinorDataVersion()) {
            return "not_actual_data";
        }
        return null;
    }
}
