package ru.yandex.direct.communication.facade.impl.actions;

import java.math.BigDecimal;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.uc.UcCampaignService;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;

import static java.util.Collections.singletonList;

@Component
public class ApplyCPMRecommendation extends AbstractApplyBudgetRecommendation {

    private static final Logger logger = LoggerFactory.getLogger(ApplyCPMRecommendation.class);

    private final GrutUacCampaignService grutUacCampaignService;
    private final UcCampaignService ucCampaignService;

    @Autowired
    public ApplyCPMRecommendation(
            CommunicationClient communicationClient,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            GrutUacCampaignService grutUacCampaignService,
            UcCampaignService ucCampaignService) {
        super(communicationClient, shardHelper, campaignRepository);
        this.grutUacCampaignService = grutUacCampaignService;
        this.ucCampaignService = ucCampaignService;
    }

    @Override
    public String getName() {
        return "APPLY_CPM";
    }

    @Override
    ActionResultImpl applyUpdatedBudget(List eventsToSend, ActionTarget target,
                                        AdditionalInfoContainer additionalInfo, Slot slot,
                                        CommunicationMessage message, CommunicationEventVersion eventVersion) {
        var shard = shardHelper.getShardByClientIdStrictly(additionalInfo.getClientId().get());
        var campaignId = target.getTargetObjectId();
        var cpm = new BigDecimal(message.getData().get("new_cpm").toString());
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, singletonList(campaignId));
        if (campaigns.isEmpty()) {
            logger.error("Campaign {} is not found", campaignId);
            return super
                    .getResult(false, campaignId, eventsToSend, target, additionalInfo, slot, eventVersion);
        }
        var campaign = campaigns.get(0);
        var strategy = campaign.getStrategy();
        strategy.getStrategyData().setAvgCpm(cpm);
        var operatorUid = message.getUserId();
        var uidAndClientId = UidAndClientId.of(campaign.getUserId(),
                ClientId.fromLong(campaign.getClientId()));
        ModelChanges<CampaignWithCustomStrategy> changes = ModelChanges.build(campaignId,
                CampaignWithCustomStrategy.class, CampaignWithCustomStrategy.STRATEGY, strategy);
        Result<Long> result = ucCampaignService.updateCampaign(changes, operatorUid, uidAndClientId).get(0);
        if (result.isSuccessful()) {
            grutUacCampaignService.updateCampaignAvgCpm(List.of(campaignId), cpm);
        }

        return super
                .getResult(result, campaignId, eventsToSend, target, additionalInfo, slot, eventVersion);
    }

}
