package ru.yandex.direct.communication.facade.impl.actions;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.facade.impl.logging.RecommendationLogger;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.uc.UcCampaignService;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider;
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.communication.CommunicationHelper.buildApplyEvent;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class ApplyUCGoalsRecommendation extends AbstractClientAction<ActionResultImpl> {

    private static final Logger logger = LoggerFactory.getLogger(ApplyUCGoalsRecommendation.class);

    private final GrutUacCampaignService grutUacCampaignService;
    private final GrutTransactionProvider grutTransactionProvider;
    private final int grutRetries;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final UcCampaignService ucCampaignService;

    @Autowired
    public ApplyUCGoalsRecommendation(
            CommunicationClient communicationClient,
            GrutUacCampaignService grutUacCampaignService,
            GrutTransactionProvider grutTransactionProvider,
            @Value("${object_api.retries}") int grutRetries,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            UcCampaignService ucCampaignService
    ) {
        super(communicationClient);
        this.grutUacCampaignService = grutUacCampaignService;
        this.grutTransactionProvider = grutTransactionProvider;
        this.grutRetries = grutRetries;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.ucCampaignService = ucCampaignService;
    }

    @Override
    public String getName() {
        return "APPLY_UC_GOALS";
    }

    @Override
    protected ActionResultImpl failByReadOnly(Slot slot, Collection<ActionTarget> targets) {
        return new ActionResultImpl(emptyList(), mapList(targets, ActionTarget::getTargetObjectId));
    }

    @Override
    protected ActionResultImpl confirmInternal(
            List eventsToSend,
            ActionTarget target,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            long buttonId,
            CommunicationMessage message,
            CommunicationEventVersion eventVersion
    ) {
        var campaignId = target.getTargetObjectId();
        if (slot.getId() == 4L /* Страница редактирования мастер-кампании */) {
            // Применять в базе не нужно. Цели подставятся в соответствующие поля на странице редактирования.
            RecommendationLogger.logApply(additionalInfo, target, slot.getId());
            var button = message.getContent().getButtons().get((int) buttonId);
            if (button.getSuccessText() != null || button.getSuccessTitle() != null) {
                return new ActionResultImpl(List.of(campaignId), emptyList(), List.of(new SubResult()
                        .withObjectId(campaignId)
                        .withTitle(button.getSuccessTitle())
                        .withText(button.getSuccessText())
                ));
            }
            return new ActionResultImpl(List.of(campaignId), emptyList());
        }
        String failReason = getFailReason(target, message, eventVersion);
        if (failReason != null) {
            RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), failReason);
            return new ActionResultImpl(emptyList(), List.of(campaignId));
        }
        Set<Long> goalIds = getGoalIdsFromData(message.getData());
        if (updateCampaignGoals(campaignId, goalIds, additionalInfo)) {
            RecommendationLogger.logApply(additionalInfo, target, slot.getId());
            eventsToSend.add(buildApplyEvent(target, eventVersion, additionalInfo));
            var button = message.getContent().getButtons().get((int) buttonId);
            if (button.getSuccessText() != null || button.getSuccessTitle() != null) {
                return new ActionResultImpl(List.of(campaignId), emptyList(), List.of(new SubResult()
                        .withObjectId(campaignId)
                        .withTitle(button.getSuccessTitle())
                        .withText(button.getSuccessText())
                ));
            }
            return new ActionResultImpl(List.of(campaignId), emptyList());
        } else {
            RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), "internal");
            return new ActionResultImpl(emptyList(), List.of(campaignId));
        }
    }

    private boolean updateCampaignGoals(
            Long cid, Set<Long> goalIds,
            AdditionalInfoContainer additionalInfo) {
        var clientId = additionalInfo.getClientId().orElse(null);
        var shard = shardHelper.getShardByClientId(clientId);
        var strategy = campaignRepository.getCampaignsWithStrategy(shard, List.of(cid)).get(0).getStrategy();
        Long goalId = MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
        List<MeaningfulGoal> meaningfulGoals = null;
        if (goalIds.size() == 1) {
            goalId = goalIds.stream().findFirst().orElseThrow();
        } else {
            var conversionValue = additionalInfo.getCurrency()
                    .map(Currency::getUcDefaultConversionValue)
                    .orElse(BigDecimal.valueOf(100));
            meaningfulGoals = mapList(goalIds, id -> new MeaningfulGoal()
                    .withGoalId(id).withConversionValue(conversionValue)
            );
            strategy.getStrategyData().setPayForConversion(false);
        }
        strategy.getStrategyData().setGoalId(goalId);

        var changes = new ModelChanges<>(cid, TextCampaign.class)
                .process(strategy, TextCampaign.STRATEGY)
                .process(meaningfulGoals, TextCampaign.MEANINGFUL_GOALS);
        MassResult<Long> massResult = ucCampaignService.updateCampaign(changes,
                additionalInfo.getUserId().orElse(null),
                UidAndClientId.of(additionalInfo.getClientUid().get(), additionalInfo.getClientId().get()));
        if (!massResult.isSuccessful()) {
            logger.info("Goals updates failed with errors " + massResult.getErrors());
            return false;
        }
        var result = massResult.get(0);
        if (!result.isSuccessful()) {
            logger.info("Goals updates failed with errors " + result.getErrors());
            return false;
        }

        grutTransactionProvider.runInRetryableTransaction(grutRetries, null, () -> {
            grutUacCampaignService.updateCampaignGoals(List.of(cid), goalIds);
            return null;
        });
        return true;
    }

    public static Set<Long> getGoalIdsFromData(Map<String, Object> data) {
        var goals = data.get("goals");
        var jsonGoals = (JsonNode) goals;
        Set<Long> goalIds = new HashSet<>();
        for (int i = 0; i < jsonGoals.size(); i++) {
            goalIds.add(jsonGoals.get(i).get("goal_id").asLong());
        }
        return goalIds;
    }
}
