package ru.yandex.direct.communication.facade.impl.actions;

import java.math.BigDecimal;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.facade.impl.logging.RecommendationLogger;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

@Component
public class AutoApplyBudgetRecommendation extends AbstractApplyBudgetRecommendation {

    private static final Logger logger = LoggerFactory.getLogger(AutoApplyBudgetRecommendation.class);

    private final CampaignOperationService campaignOperationService;

    @Autowired
    public AutoApplyBudgetRecommendation(
            CommunicationClient communicationClient,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            CampaignOperationService campaignOperationService
    ) {
        super(communicationClient, shardHelper, campaignRepository);
        this.campaignOperationService = campaignOperationService;
    }

    @Override
    public String getName() {
        return "AUTO_APPLY_BUDGET";
    }

    @Override
    ActionResultImpl applyUpdatedBudget(
            List eventsToSend,
            ActionTarget target,
            AdditionalInfoContainer additionalInfo,
            Slot slot,
            CommunicationMessage message,
            CommunicationEventVersion eventVersion
    ) {
        var recommendedWeeklyBudget = (BigDecimal) message.getData().get("new_budget");

        var campaignId = target.getTargetObjectId();
        var campaign = campaignRepository.getCampaigns(additionalInfo.getShard().get(), singleton(campaignId)).get(0);
        var strategy = campaign.getStrategy();
        if (campaign.getSource().equals(CampaignSource.UAC)) {
            // Потом поддержим для UAC
            logger.warn("Unsupported source " + campaign.getSource());
            RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), "internal");
            return new ActionResultImpl(emptyList(), List.of(campaignId));
        }
        switch (strategy.getStrategyName()) {
            case AUTOBUDGET:
            case AUTOBUDGET_AVG_CPA:
            case AUTOBUDGET_AVG_CLICK:
                strategy.getStrategyData().setSum(recommendedWeeklyBudget);
                break;
            default:
                logger.warn("Unsupported strategy " + strategy.getStrategyName());
                RecommendationLogger.logApplyFailed(additionalInfo, target, slot.getId(), "internal");
                return new ActionResultImpl(emptyList(), List.of(campaignId));
        }

        ModelChanges<CampaignWithCustomStrategy> changes = ModelChanges.build(campaignId,
                CampaignWithCustomStrategy.class, CampaignWithCustomStrategy.STRATEGY, strategy);
        var uidAndClientId = UidAndClientId.of(campaign.getUserId(),
                ClientId.fromLong(campaign.getClientId()));
        var operatorUid = additionalInfo.getUserId();
        var options = new CampaignOptions.Builder()
                .withAutoApply(true)
                .build();
        Result<Long> result = campaignOperationService.createRestrictedCampaignUpdateOperation(
                singletonList(changes), operatorUid.orElse(null), uidAndClientId, options)
                .apply()
                .get(0);
        return super
                .getResult(result, campaignId, eventsToSend, target, additionalInfo, slot, eventVersion);
    }
}
